{$MODE Delphi}
unit TempusProcs;

interface


type
  TCharSet = set of AnsiChar;
  TypeSuffixes = array[1..3] of String;

  TCacheType = record
    FileNameWithPath: String;
    AttrResult: String;                //    ( ,   )
    AttrSortResult: String;            //     ( )
    GroupName: String;                 //  
    AgeOrUptodate: String;             //  'Age'  'Update Dirs Only' (       ),
                                       //               .
    AgeOrUptodateTime: TDateTime;      // ,       AgeOrUptodate
    LastResult: shortint;              //  () ,   - ContentGetValueW   ;
                                       //     ,            
                                       //       .
    Flags: byte;                       // 1    AttrResult,
                                       // 2   AttrSortResult,
                                       // 4    GroupName,
                                       // 8    Age,
                                       // 16    Update Dirs Only
                                       // 64   ShortTTL (,       TTL  AgeOrUptodateCacheTTLForUpdatedDirs
                                       // 128   ThisIsFolder
    AgeMode: byte;                     //         Age  
  end;

const
  INIFILENAME = 'Settings.ini';

  RESULT_DATETIME_WAS_UNCHANGED = 0;
  RESULT_DATETIME_DIR_WAS_SKIPPED = 1;
  RESULT_DATETIME_WAS_CHANGED_AND_SAVED_YNG = 2;    //       
  RESULT_DATETIME_WAS_CHANGED_AND_SAVED_OLD = 3;    //       
  RESULT_DATETIME_WAS_CHANGED_AND_NOT_SAVED = 4;    //    ,     (    )

  // Files and folders attributes:
  FILE_ATTRIBUTES_NORMAL = 0;                       // Normal file. No attributes are set.
  FILE_ATTRIBUTES_READONLY = 1;                     // Read-only file
  FILE_ATTRIBUTES_HIDDEN = 2;                       // Hidden file
  FILE_ATTRIBUTES_SYSTEM = 4;                       // System file
  FILE_ATTRIBUTES_VOLUME = 8;                       // Disk drive volume label. Attribute is read-only
  FILE_ATTRIBUTES_DIRECTORY = 16;                   // Folder or directory. Attribute is read-only
  FILE_ATTRIBUTES_ARCHIVE = 32;                     // File has changed since last backup
  FILE_ATTRIBUTES_Alias = 1024;                     // Link or shortcut (file or folder). Attribute is read-only
  FILE_ATTRIBUTES_COMPRESSED = 2048;                // Compressed file. Attribute is read-only
  FILE_ATTRIBUTE_OFFLINE = 4096;                    // This attribute indicates that the file data is physically moved to offline storage
  FILE_ATTRIBUTE_NOT_CONTENT_INDEXED = 8192;        // The file or directory is not to be indexed by the content indexing service.
                                                    // (not used in Vista anymore)
  ONE_SECOND = 1/(24*60*60);

var
  IniPath: String;
  AlwaysUseFirstSuffix: Boolean = false;
  StopProcessFolders: Boolean = false;

  //    Attributes, AttributesSort, Age
  CacheArrLength: integer = 65535; // $1FFFF;                //     (  ,   ); 0   ;
  CacheArr: array of TCacheType;

  //      ini-:
  MarginForChangeDateTime: TDateTime = ONE_SECOND * 60;      // ,      
  ExcludedDirs: array of String;
  ExcludedPaths: array of String;

  // ExceptDirs
  // ExceptDrives
  // MaxMsForDir


//        (1 -> '01');
function IntToStrFixed(Num: integer): String;
function IntToStrNoZero(Num: integer): String;
//      Num (1   , 2 - , 3     ..)
function ChooseRightSuffix(Num: integer; Suffixes: TypeSuffixes): String;
function FShortName(const fn: string): string;
procedure SReplaceAll(var s: string; const sfrom, sto: string);
procedure STrimLeft(var s: string; chars: TCharSet);
procedure STrimRight(var s: string; chars: TCharSet);

function IniValueExists(Section, Item: string): Boolean;
procedure INISaveStrKey(Section, Item: string; Value: string);
function INILoadStrKey(Section, Item: string; DefaultValue: string): string;
function INILoadStrKeyWithPrefix(Section, Item: string; Prefix: string; DefaultValue: string): string;
procedure INISaveIntKey(Section, Item: string; Value: integer);
function INILoadIntKey(Section, Item: string; DefaultValue: integer): integer;
function INILoadIntKeyWithPrefix(Section, Item: string; Prefix: string; DefaultValue: integer): integer;
procedure INISaveBoolKey(Section, Item: string; Value: Boolean);
function INILoadBoolKey(Section, Item: string; DefaultValue: Boolean): Boolean;

function ToANSI(const s: Ansistring): Ansistring;
function GetFileDateTimeAndAttr(FileName: WideString;
                                WhichDateUseForFiles, WhichDateUseForDirs: integer;
                                var Attributes: cardinal;
                                var DateTime: TDateTime): Boolean;
function GetFileDateTime(const FileName: WideString; WhichDateUse: integer): TDateTime;
function GetBothDates(FileName: WideString; var CreationDateTime: TDateTime; var WriteDateTime: TDateTIme): Boolean;
function NT_SetDateTime(FolderName: WideString; dtCreation, dtLastWriteTime, dtLastAccessTime: TDateTime): Boolean;

function ProcessFolder(FolderName: WideString;
                       DepthLevels: integer;
                       var MaxFilesToCheck: integer;
                       TryToSaveCreationDateMode: integer;
                       var NewDateTime: TDateTime;
                       var DateWasChanged: integer;
                       Recursion: Boolean): Boolean;
function ExtractFilenameWithBaseDir(FileNameWithPath: String): String;

function CtrlPressed: Boolean;
function ShiftPressed: Boolean;
function AltPressed: Boolean;
function FastHashCount(FileName: String; MaxCacheNum: LongWord): LongWord;
function TrimChar(const Str: string; Ch: Char): string;


implementation
uses
  Windows,
  SysUtils,
  DateUtils,
  IniFiles;


{----------------------------------------------------------------------------------------------}

//        (1 -> '01');
function IntToStrFixed(Num: integer): String;
begin
  if (Num = 0) then
  begin
    Result := '';
    Exit;
  end;

  Result := IntToStr(Num);
  if (length(Result) = 1) then
    Result := '0' + Result;
end;

function IntToStrNoZero(Num: integer): String;
begin
  if (Num = 0) then
  begin
    Result := '';
    Exit;
  end;

  Result := IntToStr(Num);
end;

{----------------------------------------------------------------------------------------------}

//      ( , - ,    ..)
// : 1  , 2  , 3 
function ChooseRightSuffix(Num: integer; Suffixes: TypeSuffixes): String;
var
  NumMod10: integer;
  NumMod100: integer;
begin
  NumMod10 := Num mod 10;
  NumMod100 := Num mod 100;

  if (AlwaysUseFirstSuffix) or ((NumMod10 = 1) and (NumMod100 <> 11)) then
    Result := Suffixes[1]
  else
  if (NumMod10 >= 2) and (NumMod10 <= 4) and ((NumMod100 < 12) or (NumMod100 > 14)) then
    Result := Suffixes[2]
  else
    Result := Suffixes[3]
end;

{----------------------------------------------------------------------------------------------}

procedure STrimRight(var s: string; chars: TCharSet);
var
  i: integer;
begin
  i:= Length(s);
  while (i>0) and (s[i] in chars) do Dec(i);
  SetLength(s, i);
end;

{----------------------------------------------------------------------------------------------}

procedure STrimLeft(var s: string; chars: TCharSet);
var
  i: integer;
begin
  i:= 1;
  while (i<=Length(s)) and (s[i] in chars) do Inc(i);
  Delete(s, 1, i-1);
end;

{----------------------------------------------------------------------------------------------}

function ToANSI(const s: AnsiString): AnsiString;
begin
  SetLength(Result, Length(s));
  OemToCharBuffA(PAnsiChar(s), PAnsiChar(Result), Length(s));
end;

{----------------------------------------------------------------------------------------------}

procedure SReplaceAll(var s: string; const sfrom, sto: string);
var
  i: integer;
begin
  repeat
    i:= Pos(sfrom, s);
    if i=0 then Break;
    Delete(s, i, Length(sfrom));
    Insert(sto, s, i);
  until false;
end;

{----------------------------------------------------------------------------------------------}

function FShortName(const fn: string): string;
var
  buf: array[0..MAX_PATH-1] of char;
begin
  Result:= '';
  FillChar(buf, SizeOf(buf), 0);
  if GetShortPathName(PChar(fn), buf, MAX_PATH)>0 then
    Result:= buf;
end;

{----------------------------------------------------------------------------------------------}

function IniValueExists(Section, Item: string): Boolean;
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    Result := IniFile.ValueExists(Section, Item);
    FreeAndNil(IniFile);
  except

  end;
end;

procedure INISaveStrKey(Section, Item: string; Value: string);
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    try
      IniFile.WriteString(Section, Item, Value);
    finally
      FreeAndNil(IniFile);
    end;
  except

  end;
end;

function INILoadStrKey(Section, Item: string; DefaultValue: string): string;
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    Result := IniFile.ReadString(Section, Item, DefaultValue);
    FreeAndNil(IniFile);
  except

  end;
end;

//      ,        .
//    ,   DefaultValue;
function INILoadStrKeyWithPrefix(Section, Item: string; Prefix: string; DefaultValue: string): string;
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    if IniFile.ValueExists(Section, Prefix + Item) then
      Result := IniFile.ReadString(Section, Prefix + Item, DefaultValue)
    else
      Result := IniFile.ReadString(Section, Item, DefaultValue);
    FreeAndNil(IniFile);
  except

  end;
end;

procedure INISaveIntKey(Section, Item: string; Value: integer);
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    try
      IniFile.WriteInteger(Section, Item, Value);
    finally
      FreeAndNil(IniFile);
    end;
  except

  end;
end;

function INILoadIntKey(Section, Item: string; DefaultValue: integer): integer;
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    Result := IniFile.ReadInteger(Section, Item, DefaultValue);
    FreeAndNil(IniFile);
  except

  end;
end;

//      ,        .
//    ,   DefaultValue;
function INILoadIntKeyWithPrefix(Section, Item: string; Prefix: string; DefaultValue: integer): integer;
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    if IniFile.ValueExists(Section, Prefix + Item) then
      Result := IniFile.ReadInteger(Section, Prefix + Item, DefaultValue)
    else
      Result := IniFile.ReadInteger(Section, Item, DefaultValue);

    FreeAndNil(IniFile);
  except

  end;
end;

procedure INISaveBoolKey(Section, Item: string; Value: Boolean);
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    try
      IniFile.WriteBool(Section, Item, Value);
    finally
      FreeAndNil(IniFile);
    end;
  except

  end;
end;

function INILoadBoolKey(Section, Item: string; DefaultValue: Boolean): Boolean;
var
  IniFile: TIniFile;
begin
  try
    IniFile := TIniFile.Create(IniPath + INIFILENAME);
    Result := IniFile.ReadBool(Section, Item, DefaultValue);
    FreeAndNil(IniFile);
  except

  end;
end;

{----------------------------------------------------------------------------------------------}

//     /
function GetFileDateTimeAndAttr(FileName: WideString;
                                WhichDateUseForFiles, WhichDateUseForDirs: integer;
                                var Attributes: cardinal;
                                var DateTime: TDateTime): Boolean;
var
  FileSpecs: TGetFileExInfoLevels;
  FileData: TWin32FileAttributeData;
  LocalFileTime: TFileTime;
  SystemTime: TSystemTime;
  WhichDateUse: integer;

begin
  Result := False;
  DateTime := 0;
  Attributes := 0;
  FillChar(FileSpecs, SizeOf(FileSpecs), 0);
  FileSpecs := GetFileExInfoStandard;
  FillChar(FileData, SizeOf(FileData), 0);

  if GetFileAttributesExW(PWideChar(FileName), FileSpecs, @FileData) then
  begin
    //      
    if (FileData.dwFileAttributes and FILE_ATTRIBUTES_DIRECTORY) = FILE_ATTRIBUTES_DIRECTORY then
      WhichDateUse := WhichDateUseForDirs
    else
      WhichDateUse := WhichDateUseForFiles;


    if (WhichDateUse = 1) then
      FileTimeToLocalFileTime(FileData.ftCreationTime, LocalFileTime)
    else
    if (WhichDateUse = 2) then
      FileTimeToLocalFileTime(FileData.ftLastWriteTime, LocalFileTime)

    else
    if (WhichDateUse = 3) then
    begin
      if (INT64(FileData.ftLastWriteTime) > INT64(FileData.ftCreationTime)) then
        FileTimeToLocalFileTime(FileData.ftLastWriteTime, LocalFileTime)
      else
        FileTimeToLocalFileTime(FileData.ftCreationTime, LocalFileTime)
    end
    else
    begin
      if (INT64(FileData.ftLastWriteTime) < INT64(FileData.ftCreationTime)) then
        FileTimeToLocalFileTime(FileData.ftLastWriteTime, LocalFileTime)
      else
        FileTimeToLocalFileTime(FileData.ftCreationTime, LocalFileTime)
    end;

    //    ,    :
    FileTimeToSystemTime(LocalFileTime, SystemTime);
    DateTime := SystemTimeToDateTime(SystemTime);
    Attributes := FileData.dwFileAttributes;
    Result := True;
  end;
end;


function GetFileDateTime(const FileName: WideString; WhichDateUse: integer): TDateTime;
var
  Handle: THandle;
  FindData: TWin32FindDataW;
  LocalFileTime: TFileTime;
  DateTimeInt: integer;

begin
  // FileAge()
  Result := 0;
  Handle := FindFirstFileW(PWideChar(FileName), FindData);
  if Handle <> INVALID_HANDLE_VALUE then
  begin
    Windows.FindClose(Handle);
    if (FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) = 0 then
    begin
      if (WhichDateUse = 1) then
        FileTimeToLocalFileTime(FindData.ftCreationTime, LocalFileTime)
      else
      if (WhichDateUse = 2) then
        FileTimeToLocalFileTime(FindData.ftLastWriteTime, LocalFileTime)

      else
      if (WhichDateUse = 3) then
      begin
        if (INT64(FindData.ftLastWriteTime) > INT64(FindData.ftCreationTime)) then
          FileTimeToLocalFileTime(FindData.ftLastWriteTime, LocalFileTime)
        else
          FileTimeToLocalFileTime(FindData.ftCreationTime, LocalFileTime)
      end
      else
      begin
        if (INT64(FindData.ftLastWriteTime) < INT64(FindData.ftCreationTime)) then
          FileTimeToLocalFileTime(FindData.ftLastWriteTime, LocalFileTime)
        else
          FileTimeToLocalFileTime(FindData.ftCreationTime, LocalFileTime)
      end;

      
      if FileTimeToDosDateTime(LocalFileTime, LongRec(DateTimeInt).Hi, LongRec(DateTimeInt).Lo) then
      begin
        if (DateTimeInt <> -1) then
          Result := FileDateToDateTime(DateTimeInt);
      end;
    end;
  end;
end;



// Sets the time for both files and directories
// for NT
// by Nicholas Robinson
//     -     
function NT_SetDateTime(FolderName: WideString; dtCreation, dtLastWriteTime, dtLastAccessTime: TDateTime): Boolean;
var
  hDir: THandle;
  ftCreation: TFiletime;
  ftLastAccessTime: TFiletime;
  ftLastWriteTime: TFiletime;

  FileSpecs: TGetFileExInfoLevels;
  FolderData: TWin32FileAttributeData;

  function DTtoFT(DT: TDateTime): TFileTime;
  var
    LocalFileTime: TFileTime;
    SystemTime: TSystemTime;
  begin
    DateTimeToSystemTime(DT, SystemTime);
    SystemTimeToFileTime(SystemTime, LocalFileTime);
    LocalFileTimeToFileTime(LocalFileTime, Result);
  end;

begin
  //     
  FillChar(FileSpecs, SizeOf(FileSpecs), 0);
  FileSpecs := GetFileExInfoStandard;
  FillChar(FolderData, SizeOf(folderdata), 0);
  if GetFileAttributesExW(PWideChar(FolderName), FileSpecs, @FolderData) then
  begin
    //  
    ftCreation := FolderData.ftCreationTime;
    ftLastWriteTime := FolderData.ftLastWriteTime;
    ftLastAccessTime := FolderData.ftLastAccessTime;

    //   
    hDir := CreateFileW(PWideChar(FolderName),
                       GENERIC_READ or GENERIC_WRITE,
                       0,
                       nil,
                       OPEN_EXISTING,
                       FILE_FLAG_BACKUP_SEMANTICS,
                       0);
    if hDir <> INVALID_HANDLE_VALUE then
     begin
      try
        if (dtCreation <> 0) then       ftCreation :=       DTtoFT(dtCreation);
        if (dtLastAccessTime <> 0) then ftLastAccessTime := DTtoFT(dtLastAccessTime);
        if (dtLastWriteTime <> 0) then  ftLastWriteTime  := DTtoFT(dtLastWriteTime);
        Result := SetFileTime(hDir, @ftCreation, @ftLastAccessTime, @ftLastWriteTime);
      finally
        CloseHandle(hDir);
      end;
    end
    else
      Result := False;
  end
  else
    //      
    Result := False;
end;


//     / (  )
function GetBothDates(FileName: WideString; var CreationDateTime: TDateTime; var WriteDateTime: TDateTIme): Boolean;
var
  FileSpecs: TGetFileExInfoLevels;
  FileData: TWin32FileAttributeData;
  LocalFileTime: TFileTime;
  SystemTime: TSystemTime;

begin
  Result := False;
  CreationDateTime := 0;
  WriteDateTime := 0;
  FillChar(FileSpecs, SizeOf(FileSpecs), 0);
  FileSpecs := GetFileExInfoStandard;
  FillChar(FileData, SizeOf(FileData), 0);

  if GetFileAttributesExW(PWideChar(FileName), FileSpecs, @FileData) then
  begin
    FileTimeToLocalFileTime(FileData.ftCreationTime, LocalFileTime);
    FileTimeToSystemTime(LocalFileTime, SystemTime);
    CreationDateTime := SystemTimeToDateTime(SystemTime);

    FileTimeToLocalFileTime(FileData.ftLastWriteTime, LocalFileTime);
    FileTimeToSystemTime(LocalFileTime, SystemTime);
    WriteDateTime := SystemTimeToDateTime(SystemTime);

    Result := True;
  end;
end;


{----------------------------------------------------------------------------------------------}

//  :   ,         
//  True,    .
// DepthLevels       . 1    ;
// MaxFilesToCheck       (    );
// TryToSaveCreationDateMode  0..2    3      (.  TryToSaveCreationDate);
// MarginForChangeDateTime                 ;
// DateWasChanged = True,       ;
// Recursion  True    , False  
function ProcessFolder(FolderName: WideString;
                       DepthLevels: integer;
                       var MaxFilesToCheck: integer;
                       TryToSaveCreationDateMode: integer;
                       var NewDateTime: TDateTime;
                       var DateWasChanged: integer;
                       Recursion: Boolean): Boolean;

  //      ExcludedDirs
  function FindNameInExcludedDirs(Name: string): Boolean;
  var i: integer;
  begin
    // TODO:      ,     / (- GetGroupNumByName  Tempus.pas)
    Result := false;
    for i := 0 to length(ExcludedDirs)-1 do
      if ExcludedDirs[i] = Name then
      begin
        Result := True;
        Break;
      end;
  end;


  //          ()
  procedure ResetCacheForDirAndParentDirs(FolderName: String);
  var
    TempCacheNum: LongWord;
    ParentFolderName: String;

  begin
    TempCacheNum := FastHashCount(ExtractFileNameWithBaseDir(FolderName), CacheArrLength);
    if CacheArr[TempCacheNum].FileNameWithPath = FolderName then
      CacheArr[TempCacheNum].AgeOrUptodateTime := 0;

    ParentFolderName := ExtractFileDir(FolderName);
    //         
    if length(ParentFolderName) > 3 then
      ResetCacheForDirAndParentDirs(ParentFolderName);
  end;


var
  ThisDirCreationTime: TDateTime;
  ThisDirWriteTime: TDateTime;

  Hour_, Min_, Sec_, Milli_: Word;
  SearchRec: TSearchRec;
  Res: Integer;
  MaxDateTimeFoundInFolder: TDateTime;
  FoundFileDateTime: TDateTime;
  FoundDirDateWasChanged: integer;

  WasBreak: Boolean;
  i: integer;

begin
  if (DepthLevels = 0) or (MaxFilesToCheck = 0) then
  begin
    DateWasChanged := RESULT_DATETIME_DIR_WAS_SKIPPED;
    Result := True;
    Exit;
  end;

  // ,         
  for i := 0 to length(ExcludedPaths)-1 do
    if AnsiPos(ExcludedPaths[i], AnsiLowerCase(AnsiString(FolderName))) > 0 then
    begin
      DateWasChanged := RESULT_DATETIME_DIR_WAS_SKIPPED;
      Result := True;
      Exit;
    end;

  if (not Recursion) and
     FindNameInExcludedDirs(AnsiLowerCase(ExtractFileName(FolderName))) then
  begin
    DateWasChanged := RESULT_DATETIME_DIR_WAS_SKIPPED;
    Result := True;
    Exit;
  end;


  // Debug:
  // Sleep(100);
  // Windows.Beep(200, 30);

  DateWasChanged := RESULT_DATETIME_WAS_UNCHANGED;
  try
    if GetBothDates(FolderName, ThisDirCreationTime, ThisDirWriteTime) then
    begin
      // (1)      ''   
      // (     ,      )
      if (ThisDirCreationTime > ThisDirWriteTime) and
         (TryToSaveCreationDateMode > 0) then
      begin
        DecodeTime(ThisDirCreationTime, Hour_, Min_, Sec_, Milli_);

        // ,  ''       
        if not ((Sec_ mod 10 = 0) and (Milli_ = 700)) then
        begin
          //  :       :
          ThisDirCreationTime := ThisDirWriteTime;
          NT_SetDateTime(FolderName, ThisDirCreationTime, 0, 0);
        end;
      end;


      // (2)      
      WasBreak := False;
      MaxDateTimeFoundInFolder := 0;

      // )      
      // ( ,       -  MaxFilesToCheck        )
      Res := FindFirst(FolderName + '\*', faAnyFile - faDirectory, SearchRec);
      while Res = 0 do
      begin
        Dec(MaxFilesToCheck);
        if (MaxFilesToCheck = 0) or (StopProcessFolders) then
          begin WasBreak := true; Break; end;

        // (!)    FindFirst/FindNext   :  ,      - ,
        //      .
        FoundFileDateTime := FiledateToDateTime(SearchRec.Time);
        if FoundFileDateTime > MaxDateTimeFoundInFolder then
          MaxDateTimeFoundInFolder := FoundFileDateTime;

        Res := FindNext(SearchRec);
      end;
      FindClose(SearchRec);


      // )        
      if not WasBreak then
      begin
        Res := FindFirst(FolderName + '\*', faAnyFile, SearchRec);
        while Res = 0 do
        begin
          if (SearchRec.Attr and FILE_ATTRIBUTES_DIRECTORY <> 0) and
             (SearchRec.Name <> '.') and
             (SearchRec.Name <> '..') then
          begin
            Dec(MaxFilesToCheck);
            if (MaxFilesToCheck = 0) or (StopProcessFolders) then
              begin WasBreak := true; Break; end;

            //        
            if FindNameInExcludedDirs(AnsiLowerCase(SearchRec.Name)) then
            begin
              Res := FindNext(SearchRec);
              Continue;
            end;

            if DepthLevels > 1 then
            begin
              FoundFileDateTime := FiledateToDateTime(SearchRec.Time);
              //    
              ProcessFolder(FolderName + '\' + SearchRec.Name,
                            DepthLevels-1,
                            MaxFilesToCheck,
                            TryToSaveCreationDateMode,
                            FoundFileDateTime,
                            FoundDirDateWasChanged,
                            True);
            end
            else
              //    
              FoundFileDateTime := FiledateToDateTime(SearchRec.Time);

            if FoundFileDateTime > MaxDateTimeFoundInFolder then
              MaxDateTimeFoundInFolder := FoundFileDateTime;
          end;

          Res := FindNext(SearchRec);
        end;
        FindClose(SearchRec);
      end;


      
      // (3)      / ,   
      if (MaxDateTimeFoundInFolder <> 0) and
         (Abs(ThisDirWriteTime - MaxDateTimeFoundInFolder) > MarginForChangeDateTime) and
         //      ,         
         ((not WasBreak) or (MaxDateTimeFoundInFolder - ThisDirWriteTime > MarginForChangeDateTime)) then
      begin

        //       ,    .
        //  ,       TryToSaveCreationDateMode
        if (MaxDateTimeFoundInFolder < ThisDirCreationTime) and
           (TryToSaveCreationDateMode > 0) then
        begin
          if (TryToSaveCreationDateMode = 1) then
            // 1)       
            MaxDateTimeFoundInFolder := ThisDirCreationTime + ONE_SECOND
          else
          if (TryToSaveCreationDateMode = 2) then
          begin
            // 2)      
            DecodeTime(ThisDirCreationTime, Hour_, Min_, Sec_, Milli_);
            ThisDirCreationTime := RecodeTime(ThisDirCreationTime, Hour_, Min_, Sec_ - (Sec_ mod 10), 700);
            NT_SetDateTime(FolderName, ThisDirCreationTime, 0, 0);
          end;
        end;

        //   / 
        NewDateTime := MaxDateTimeFoundInFolder;

        //     
        if NT_SetDateTime(FolderName, 0, MaxDateTimeFoundInFolder, 0) then
        begin
          if MaxDateTimeFoundInFolder > ThisDirWriteTime then
            DateWasChanged := RESULT_DATETIME_WAS_CHANGED_AND_SAVED_YNG
          else
            DateWasChanged := RESULT_DATETIME_WAS_CHANGED_AND_SAVED_OLD;

          //             
          // (TODO: -      ,       DontRecalcThisDirNextTime,
          // ..              )
          if (Length(CacheArr) > 0) then
            ResetCacheForDirAndParentDirs(FolderName);
        end
        else
          DateWasChanged := RESULT_DATETIME_WAS_CHANGED_AND_NOT_SAVED;
      end
      else
      begin
        //   / 
        NewDateTime := ThisDirWriteTime;
        DateWasChanged := RESULT_DATETIME_WAS_UNCHANGED;
      end;

      Result := True;
    end;

  except
    Result := False;
    NewDateTime := 0;
  end;
end;


{----------------------------------------------------------------------------------------------}

//        (, 'Folder\Filename.exe')
function ExtractFilenameWithBaseDir(FileNameWithPath: String): String;
var
  i: integer;
  LevelsCounter: integer;
begin
  LevelsCounter := 2;
  for i := length(FileNameWithPath) downto 1 do
    if (FileNameWithPath[i] = '\') then
    begin
      dec(LevelsCounter);
      if (LevelsCounter = 0) then
        break;
    end;

  if LevelsCounter > 0 then
    Result := FileNameWithPath
  else
    Result := Copy(FileNameWithPath, i+1, MaxInt);
end;

{----------------------------------------------------------------------------------------------}


function CtrlPressed: Boolean;
var
  State: TKeyboardState;
begin
  GetKeyboardState(State);
  Result := ((State[vk_Control] and 128) <> 0);

  // VK_LSHIFT
  // VK_RSHIFT
  // VK_LCONTROL
  // VK_RCONTROL
  // VK_LMENU
  // VK_RMENU
  // VK_CAPITAL  Caps Lock
  // VK_PAUSE  Pause/Break key
  // VK_NUMLOCK
  // VK_SCROLL  Scroll Lock
  // http://delphi.about.com/od/objectpascalide/l/blvkc.htm
end;

function ShiftPressed: Boolean;
var
  State: TKeyboardState;
begin
  GetKeyboardState(State);
  Result := ((State[vk_Shift] and 128) <> 0);
end;

function AltPressed: Boolean;
var
  State: TKeyboardState;
begin
  GetKeyboardState(State);
  Result := ((State[vk_Menu] and 128) <> 0);
end;

{----------------------------------------------------------------------------------------------}

function FNV1aHash(const s: AnsiString): LongWord;
var
    i: Integer;
const
    FNV_offset_basis = 2166136261;
    FNV_prime = 16777619;
begin
    //FNV-1a hash
    Result := FNV_offset_basis;
    for i := 1 to Length(s) do
        Result := (Result xor Byte(s[i])) * FNV_prime;
end;

//    
function FastHashCount(FileName: String; MaxCacheNum: LongWord): LongWord;
var
  Tmp: LongWord;
begin
  {Result := 0;
  for i:=1 to length(FileName) do
    Result := ((Result xor byte(FileName[i])) shl 1) or (Result and $8000) shr 15;
  Result := Result mod MaxCacheNum;}

  Tmp := FNV1aHash(FileName);
  Result := Tmp mod MaxCacheNum;
end;

{----------------------------------------------------------------------------------------------}

//         
function TrimChar(const Str: string; Ch: Char): string;
var
  S, E: integer;
begin
  S := 1;
  while (S <= Length(Str)) and (Str[S] = Ch) do Inc(S);
  E := Length(Str);
  while (E >= 1) and (Str[E] = Ch) do Dec(E);
  if (S > E) then
    Result := ''
  else
    Result := Copy(Str, S, E - S + 1);
  // SetString(Result, PChar(@Str[S]), E - S + 1);
end;

{----------------------------------------------------------------------------------------------}

end.
