unit uKCFile;

{ History:
	07.06.2004	Setzen von blnr bei RAW-Dateien, damit die erzeugten
			KCC-Files von Haftmann-Emulator erkannt werden
	19.03.2006	und das war die falsche Speicherzelle. Richtig ist lblnr
  13.06.2007  Umstellung auf Class, Korrektur Vendor-Erkennung, neue Viewer fr SCRIPT, HEX
  04.09.2008  Zeile 163:  bei KCC-Dateien gab es teilw. einen zustzlichen kopfblock
      wg. KCBinaryFile[i] < 33 statt < 32
  18.02.2011  Basic Arrays (TTT) werden nun auch ausgewertet
  07.10.2012  gebastelten Kopfblock als "virtuell" anzeigen

  30.10.2018  bei robotron BASIC-Dateien als .ZBS (mit 11 Byte Kopf) statt .SSS (ohne 11 Byte Kopf) speichern

  17.04.2020  endlich den Speicherfehler gefunden (bei BASIC-Programme beginnen mit Block 1. Hier wurde ber das Ende hinaus korrigiert)
  	      Erkennung von ftMIT_BLOCKNR nun rigoroser, 128*129*x=16512, 33024, 49536 wird nun ftOHNE_BLOCKNR)
  }

interface

uses SysUtils, Classes;

type
  TKCFileType=(ftUNKNOWN, ftAF_TAPE, ftMIT_BLOCKNR, ftOHNE_BLOCKNR, ftRAW, ftBASIC);
  TKCData=(UNDEF, UNKNOWN, BASIC_CSAVE, BASIC_ARRAY, BASIC_LIST, ASM_EDAS, FORTH, TEXT1, SCRIPT, TEXT_ASM, BASICODE, BASIC_KCB);
  TKCVendor=(vUNKNOWN, vROBOTRON, vMUEHLHAUSEN);

  TKCBinaryFile = packed array of byte;

type
  { FileControlBlock Standard-Z9001-Programm }
  TFCBMC = packed record
    dateiname: array[0..7] of char;
    dateityp: array[0..2] of char;
    e1, e2: byte;
    psum, arb, blnr, lblnr: byte;
    aadr, eadr, sadr: word;
    sby: byte;
  end;

  { FileControlBlock Basic-Programm }
  TFCBBASIC = packed record
    dateityp: array[0..2] of char;
    dateiname: array[0..7] of char;
    adr1, adr2: word;
  end;

const
  AFHEADER = 'KC-TAPE by AF. ';

type
  TKCFile = class
  public
    //KCFileHeader: TKCFileHeader;
    //filesize: integer;
    origKCFileType: TKCFileType;
    KCFileType: TKCFileType;
    Filetyp: string[30];      // Klartext
    KCData: TKCData;

    filename: string[8];
    fileext: string[3];
    aadr, eadr, sadr: word;
    first_blocknr: byte;
    Vendor: TKCVendor;

    IgnoreBASICEnd: boolean;
    VirtualHeader: boolean;
  private
    KCBinaryFile: TKCBinaryFile;       // mit Blocknr
    KCBinFile: TKCBinaryFile;          // ohne Blocknr
    function BasicToken(c: char; KCVendor: TKCVendor = vROBOTRON): string;
    procedure ConvertASCIIBasic(strings: TStrings);
    procedure ConvertText(strings: TStrings; offset: integer = 128);
    procedure ConvertBasicode(strings: TStrings);
    procedure ConvertForth(strings: TStrings);
    procedure ConvertText1(strings: TStrings);
    procedure ConvertTextAll(strings: TStrings);
    procedure ConvertScript(strings: TStrings);
    procedure ConvertHex(strings: TStrings);
    procedure ConvertBasic(strings: TStrings; KCVendor: TKCVendor = vROBOTRON; offset: integer = 13);
    procedure ConvertBasicKCB(strings: TStrings);
    procedure ConvertBasicArray(strings: TStrings);
    procedure UpdateFCB;
    procedure GetKCData;
  public
    procedure LoadFile(filename: string; var typ: TKCFileType);
    procedure SaveFile(filename: string; typ: TKCFileType);
    procedure DecodeKCFile(strings: TStrings; typ: TKCData = UNDEF);
  end;

(*
  // Multi-AF-File
  TAFFile = class
  public
    filesize: integer;
    KCFiles: array of TKCFile;
    procedure LoadFile(filename: string; var typ: TKCFileType);
    procedure SaveFile(filename: string; typ: TKCFileType);
  end;
*)

implementation

uses Math, Dialogs;

procedure TKCFile.LoadFile(filename: string; var typ: TKCFileType);
var
  f: TFileStream;
  header: array[0..16] of char;
  filesize: integer;
  i, blockanzahl, blockstart: integer;
  s, s1: string;
  file_ext: string;
begin
  // KC-FileInfo leeren
  filetyp := '';
  KCFileType := ftUNKNOWN;
  KCData := UNKNOWN;

  fileext := '';
  aadr := 0; eadr := 0; sadr := 0;
  first_blocknr := 0;
  Vendor := vROBOTRON;

  KCBinaryFile := nil;
  KCBinFile := nil;

  file_ext := UpperCase(ExtractFileExt(filename));
  f := TFileStream.Create(filename, fmOpenRead);
  try
    filesize := f.size;

    // Typbestimmung
    f.Read(header, 16); header[16] := #0;
    if header = AFHEADER then begin
      typ := ftAF_TAPE;
      filesize := filesize - length(AFHEADER);
      if filesize mod 129 <> 0 then
        //raise EFilerError.Create('ungltiges AF-Tape-Format (Lnge stimmt nicht)');
        MessageDlg('ungltiges AF-Tape-Format (Lnge stimmt nicht)', mtWarning, [mbOK], 0);
    end else if ((filesize mod 129 = 0) and (filesize mod 128 <> 0)) then begin
      typ := ftMIT_BLOCKNR;
      f.seek(0, soFromBeginning);
    end else begin
      typ := ftOHNE_BLOCKNR;
      f.seek(0, soFromBeginning);
    end;

    // Einlesen der Datei
    SetLength(KCBinaryFile,filesize);
    i := f.read(KCBinaryFile[0], filesize);
    if i <> filesize then
      raise EFilerError.Create('Datei wurde nicht komplett gelesen');
  finally
    f.free;
  end;

   SetLength(KCBinaryFile,filesize);

  // Basic-Dateien ohne Dateiname im Kopf (*.sss)
  if ((file_ext = '.SSS') or (file_ext = '.TTT')) and
  not ((KCBinaryFile[0] >= $D3) and (KCBinaryFile[0] <= $D9)
       and (KCBinaryFile[0] = KCBinaryFile[1]) and (KCBinaryFile[0] = KCBinaryFile[2]))
  then begin
    { Filename erzeugen: 3 Byte Type+$80, dann 8 Byte Name }
    s := UpperCase(ExtractFileName(filename));
    i := pos('.',s);
    s1 := char(ord(s[i+1])+$80); s1 := s1+s1+s1;
    s := copy(s,1,min(i-1, 8));
    s := s1+copy(s+'        ',1,8);
    { Filename vor Programm schreiben }
    filesize := filesize + 11;
    SetLength(KCBinaryFile,filesize);
    move(KCBinaryFile[0], KCBinaryFile[11], filesize-11);
    move(s[1],KCBinaryFile[0],11);
  end else if Typ = ftOHNE_BLOCKNR then begin
    //KCC von Mhlhausen (Mario Leubner 080909)
    {   Eine MC-Datei (*.KCC) hat zwar einen definierten Vorblock, davon sind jedoch
  	nur folgende Bytes garantiert:
	 Byte 10h - Anzahl der Argumente (kann 2-10 sein)
	 Byte 11h/12h - Startadresse (Argument 1)
	 Byte 13h/14h - Endadresse (Argument 2)
	 Byte 15h/16h - Startadresse (Argument 3) NUR GLTIG wenn
	Anzahlgrer als 2 ist.
	 Byte 17h/18h - Argument 4 falls Anzahl grer als 3 ist
	 usw. bis Argument 10.
	Definiert, aber nicht zwingend mit gltigem Inhalt belegt sind auerdem noch:
	 Byte 0-8 Dateiname
	 Byte 9-0bh Dateityp }
    if (file_ext = '.KCC')
    and (KCBinaryFile[16] >=2) and (KCBinaryFile[16] <= 10)
    then begin
	//bleibt Typ = ftOHNE_BLOCKNR
	Vendor := vMUEHLHAUSEN;
    end else
    // Mhlhausen KCB
    {	Die besagten KCC-Dateien gibt es noch als Variante KCB, das letze B
	steht hier fr Basic.  Es sind IMMER Basic-Abzge, meist selbststartende
	Spiele.
	Erzeugt werden KCB-Dateien mit einem Hilfsprogramm BSAVE, das
	automatisch die BASIC-Arbeitszellen eines im Speicher befindlichen
	Programms ausliest, eine Selbststartroutine hinzufgt und gemeinsam
	abspeichert.  Die KCB-Dateien sind also identisch mit den KCC, haben den
	gleichen Vorblock.  Anfangsadresse ist dabei meist 300h, Endadresse je
	nach Gre des BASIC-Programms (max. 7FFFh) und Startadresse ist z.B.
	0365h (der BASIC-Editierpuffer mit der Autostartroutine).
	P.S.: Das BASIC-Programm beginnt an der Adresse, die auf 035Fh/0360h
	steht. Das ist normalerweise 0401h, kann aber bei integriertem
	Maschinencode auch einmal davon abweichen. Du knntest also bei den
	KCB-Dateien diese Speicherzelle als Beginn des BASIC-Textes auswerten. }
    if (file_ext = '.KCB')
    and (KCBinaryFile[16] >=2) and (KCBinaryFile[16] <= 10)
    then begin
	//bleibt Typ = ftOHNE_BLOCKNR
	KCData := BASIC_KCB;
	Vendor := vMUEHLHAUSEN;
    end else
    //Test auf FCB (BASIC und MC) (Z9001)
    if not ((KCBinaryFile[0] >= $D3) and (KCBinaryFile[0] <= $D9)
    and (KCBinaryFile[0] = KCBinaryFile[1]) and (KCBinaryFile[0] = KCBinaryFile[2]))
    then
      for i := 0 to 10 do
        if (KCBinaryFile[i] <> 0) and ((KCBinaryFile[i] < 32) or (KCBinaryFile[i] > 127)) then
          typ := ftRAW;
  end;

  if (typ = ftRAW) then begin
    { Kopfblock basteln }
    filesize := filesize + 128;
    SetLength(KCBinaryFile,filesize);
    Move(KCBinaryFile[0], KCBinaryFile[128], filesize-128);
    FillChar(KCBinaryFile[0],128,0);
    { Filename erzeugen: 8 Byte Name + 3 Byte Typ}
    s := UpperCase(ExtractFileName(filename));
    i := pos('.',s);
    if i > 0 then s1 := copy(s+#0#0#0,i+1,3) else s1 := #0#0#0;
    s := copy(s,1,min(i-1, 8));
    s := copy(s+#0#0#0#0#0#0#0#0,1,8);
    { Filename in FCB schreiben }
    with TFCBMC(Pointer(KCBinaryFile)^) do begin
      move(s[1],dateiname,8);
      move(s1[1],dateityp,3);
      aadr := $300;
      eadr := aadr + filesize-128;
      sadr := $ffff;

      lblnr := 3;  { fr Haftmann-Emulator. Dieser interpretiert dieses Feld als Anzahl
      		    der Argumente aadr eadr sadr }
    end;
    VirtualHeader := true;
  end;

   // Blocknummern ergnzen bei RAW und OHNE_BLOCKNR
  if (typ = ftRAW) or (typ = ftOHNE_BLOCKNR) then begin
    blockanzahl := (filesize+127) div 128;
    filesize := blockanzahl*129;
    SetLength(KCBinaryFile,filesize);

    for i := 0 to blockanzahl-1 do begin
      blockstart := i*129;
      Move(KCBinaryFile[blockstart], KCBinaryFile[blockstart+1], filesize-blockstart-1);
      KCBinaryFile[blockstart] := i; { Blocknummern ab 0 fortlaufend }
    end;
    KCBinaryFile[(blockanzahl-1) * 129] := $FF; { letzter Block auf FF }
  end;

  //Korrektur auf volle Blcke
  if filesize mod 129 <> 0 then begin
    filesize := filesize + (129 - filesize mod 129);
    SetLength(KCBinaryFile, filesize);
  end;

  //Details holen
  if (KCBinaryFile[1] >= $D3) and (KCBinaryFile[1] <= $D9)
  and (KCBinaryFile[1] = KCBinaryFile[2]) and (KCBinaryFile[1] = KCBinaryFile[3])
  then with TFCBBASIC(Pointer(@KCBinaryFile[1])^) do begin
    // Basic- Programm
    self.filetyp := 'BASIC';
    self.filename := TrimRight(String(dateiname));
    self.fileext := Chr(ord(dateityp[0])-$80)+Chr(ord(dateityp[1])-$80)+Chr(ord(dateityp[2])-$80);
    self.first_blocknr := KCBinaryFile[0];
    self.aadr := $401;
    self.eadr := aadr + (KCBinaryFile[12] + KCBinaryFile[13] * $100);
    self.sadr := 0;
  end else with TFCBMC(Pointer(@KCBinaryFile[1])^) do begin
    self.filetyp := 'MC';
    self.filename := TrimRight(String(dateiname));
    self.fileext := TrimRight(String(dateityp));
    self.aadr := aadr;
    self.eadr := eadr;
    self.sadr := sadr;
    self.first_blocknr := blnr;

    if (Vendor = vMUEHLHAUSEN) then
      self.first_blocknr := 0;
  end;

  // BASIC-Programme beginnen mit Block 1
  if (Typ = ftOHNE_BLOCKNR) and (filetyp = 'BASIC') then begin
    for i := 0 to (filesize div 129)-1 do
      KCBinaryFile[i*129] := i+1;
    first_blocknr := KCBinaryFile[0];
  end;

  // KCBinFile: ohne Blocknr
  SetLength(KCBinFile, filesize - filesize mod 129);
  for i := 0 to High(KCBinaryFile) div 129 do
    Move(KCBinaryFile[i*129+1], KCBinFile[i*128], 128);

  origKCFileType := typ;

  // Typ ermitteln
  GetKCData;
end;


procedure TKCFile.UpdateFCB;
var
  s: String;
  i: Integer;
begin
  if filetyp = 'BASIC' then
    with TFCBBASIC(Pointer(@KCBinaryFile[1])^) do begin
      s := filename+'        '; move(s[1], dateiname, 8);
      s := fileext+'   '; move(s[1], dateityp, 3);
      for i := 0 to 2 do dateityp[i] := chr(ord(dateityp[i]) + $80);
    end
  else if filetyp = 'MC' then
     with TFCBMC(Pointer(@KCBinaryFile[1])^) do begin
      s := filename+#0#0#0#0#0#0#0#0; move(s[1], dateiname, 8);
      s := fileext+#0#0#0; move(s[1], dateityp, 3);
      aadr := self.aadr;
      eadr := self.eadr;
      sadr := self.sadr;
    end;
end;

procedure TKCFile.SaveFile(filename: string; typ: TKCFileType);
var
  f: TFileStream;
  i: Integer;
begin
  f := TFileStream.Create(filename, fmCreate or fmOpenWrite);
  case typ of
    ftAF_TAPE: begin
               f.Write(AFHEADER, Length(AFHEADER));
               f.Write(Pointer(KCBinaryFile)^, Length(KCBinaryFile));
             end;
    ftMIT_BLOCKNR: begin
               f.Write(Pointer(KCBinaryFile)^, Length(KCBinaryFile));
             end;
    ftOHNE_BLOCKNR: begin
               for i := 0 to High(KCBinaryFile) div 129 do
                 f.Write(Pointer(@KCBinaryFile[i*129+1])^, 128);
             end;
    ftBASIC: begin
               for i := 0 to High(KCBinaryFile) div 129 do
                 if (i = 0) 
                 and (Vendor=vMUEHLHAUSEN) // 30.10.2018
                 then
                   // fr Haftmann-Emu kein Filename am Anfang
                   f.Write(Pointer(@KCBinaryFile[1+11])^, 128-11)
                 else
                   f.Write(Pointer(@KCBinaryFile[i*129+1])^, 128);
             end;
  end;
  f.Free;
end;

function TKCFile.BasicToken(c: char; KCVendor: TKCVendor = vROBOTRON): string;
begin
  result := c;
  case c of
    #$80: result := 'END';
    #$81: result := 'FOR';
    #$82: result := 'NEXT';
    #$83: result := 'DATA';
    #$84: result := 'INPUT';
    #$85: result := 'DIM';
    #$86: result := 'READ';
    #$87: result := 'LET';
    #$88: result := 'GOTO';
    #$89: result := 'RUN';
    #$8a: result := 'IF';
    #$8b: result := 'RESTORE';
    #$8c: result := 'GOSUB';
    #$8d: result := 'RETURN';
    #$8e: result := 'REM';
    #$8f: result := 'STOP';
    #$90: result := 'OUT';
    #$91: result := 'ON';
    #$92: result := 'NULL';
    #$93: result := 'WAIT';
    #$94: result := 'DEF';
    #$95: result := 'POKE';
    #$96: result := 'DOKE';
    #$97: result := 'AUTO';
    #$98: result := 'LINES';
    #$99: result := 'CLS';
    #$9a: result := 'WIDTH';
    #$9b: result := 'BYE';
    #$9c: result := '!';
    #$9d: result := 'CALL';
    #$9e: result := 'PRINT';
    #$9f: result := 'CONT';
    #$a0: result := 'LIST';
    #$a1: result := 'CLEAR';
    #$a2: result := 'CLOAD';
    #$a3: result := 'CSAVE';
    #$a4: result := 'NEW';
    #$a5: result := 'TAB(';
    #$a6: result := 'TO';
    #$a7: result := 'FN';
    #$a8: result := 'SPC(';
    #$a9: result := 'THEN';
    #$aa: result := 'NOT';
    #$ab: result := 'STEP';
    #$ac: result := '+';
    #$ad: result := '-';
    #$ae: result := '*';
    #$af: result := '/';
    #$b0: result := '^';
    #$b1: result := 'AND';
    #$b2: result := 'OR';
    #$b3: result := '>';
    #$b4: result := '=';
    #$b5: result := '<';
    #$b6: result := 'SGN';
    #$b7: result := 'INT';
    #$b8: result := 'ABS';
    #$b9: result := 'USR';
    #$ba: result := 'FRE';
    #$bb: result := 'INP';
    #$bc: result := 'POS';
    #$bd: result := 'SQR';
    #$be: result := 'RND';
    #$bf: result := 'LN';
    #$c0: result := 'EXP';
    #$c1: result := 'COS';
    #$c2: result := 'SIN';
    #$c3: result := 'TAN';
    #$c4: result := 'ATN';
    #$c5: result := 'PEEK';
    #$c6: result := 'DEEK';
    #$c7: result := 'PI';
    #$c8: result := 'LEN';
    #$c9: result := 'STR$';
    #$ca: result := 'VAL';
    #$cb: result := 'ASC';
    #$cc: result := 'chr$';
    #$cd: result := 'LEFT$';
    #$ce: result := 'RIGHT$';
    #$cf: result := 'MID$';
    #$d0: result := 'LOAD';
    #$d1: result := 'TRON';
    #$d2: result := 'TROFF';
    #$d3: result := 'EDIT';
    #$d4: result := 'ELSE';

    // Erweiterung M511
    #$d5: result := 'INKEY$';
    #$d6: result := 'JOYST';
    #$d7: result := 'STRING$';
    #$d8: result := 'INSTR';
    #$d9: result := 'RENUM';	//RENUMBER
    #$da: result := 'DELETE';
    #$db: result := 'PAUSE';
    #$dc: result := 'BEEP';
    #$dd: result := 'WINDOW';
    #$de: result := 'BORDER';
    #$df: result := 'INK';
    #$e0: result := 'PAPER';
    #$e1: result := 'AT';
  end;

  if KCVendor = vROBOTRON then begin
    case c of
      //Erweiterung BM608
      #$e2: result := 'PSET';
      #$e3: result := 'LINE';
      #$e4: result := 'CIRCLE';
      #$e5: result := '!';
      #$e6: result := 'PAINT';
      #$e7: result := 'LABEL';
      #$e8: result := 'SIZE';
      #$e9: result := 'ZERO';
      #$ea: result := 'HOME';
      #$eb: result := '!';
      #$ec: result := 'GCLS';
      #$ed: result := 'SCALE';
      #$ee: result := 'SCREEN';
      #$ef: result := 'POINT';
      #$f0: result := 'XPOS';
      #$f1: result := '!';
      #$f2: result := 'YPOS';
    end;
  end else begin
    // KC 85/2-4
    case c of
      #$e2: result := 'COLOR';
      #$e3: result := 'SOUND';
      #$e4: result := 'PSET';
      #$e5: result := 'PRESET';
      #$e6: result := 'BLOAD';
      #$e7: result := 'VPEEK';
      #$e8: result := 'VPOKE';
      #$e9: result := 'LOCATE';
      #$ea: result := 'KEYLIST';
      #$eb: result := 'KEY';
      #$ec: result := 'SWITCH';
      #$ed: result := 'PTEST';
      #$ee: result := 'CLOSE';
      #$ef: result := 'OPEN';
      #$f0: result := 'RANDOMIZE';
      #$f1: result := 'VGET$';
      #$f2: result := 'LINE';
      #$f3: result := 'CIRCLE';
      #$f4: result := 'CSRLIN';
	//caos 4.7 07.12.2017       
      #$f5: result := 'DEVICE';
      #$f6: result := 'FILES';
      #$f7: result := 'CHDIR';
    end;
  end;
  if (c = #$e5) or (c = #$eb) or (c = #$f1) or (c > #$f2)
    then Vendor := vMUEHLHAUSEN;
end;

procedure TKCFile.ConvertBasic(strings: TStrings; KCVendor: TKCVendor = vROBOTRON; offset: integer = 13);
var
  i: integer;
  addrnl: integer;
  znr: integer;
  s: String;
  c: char;
type
  tstate = (st_end, st_normal, st_string);
var
  state: tstate;  
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopf berspringen (default 13 = 'D3D3D3' + 8 Byte Name + 2 Byte Gesamtlnge)
  i := offset;

  while i<length(KCBinFile) do begin
    // 2 Byte Adresse nchste Zeile
    addrnl := KCBinFile[i] + 256*KCBinFile[i+1];
    inc(i,2);
    if not IgnoreBASICEnd then
      if addrnl = 0 then break;

    //Zeilennummer
    znr := KCBinFile[i] + 256*KCBinFile[i+1];
    inc(i,2);

    //Zeile bearbeiten
    s := '';
    state := st_normal;
    while state <> st_end do begin
      c := chr(KCBinFile[i]);
      inc(i);
      case c of
      	#0: state := st_end;     // Zeilenendezeichen (0)
      	'"': if state = st_string then state := st_normal
      	     else if state = st_normal then state := st_string;
      end;
      if (state = st_normal) and (c >= #$80) then
        s := s + BasicToken(c, KCVendor)
      else if state in [st_normal, st_string] then
        s := s + c;
    end;

{$ifdef debug}
    strings.Add(Format('%.4x %d %s', [addrnl, znr, s]));
{$else}
    strings.Add(Format('%d %s', [znr, s]));
{$endif}
  end;

  Strings.EndUpdate;
end;

procedure TKCFile.ConvertBasicKCB(strings: TStrings);
var
  start: integer;
begin
  start := KCBinFile[$035F-aadr+$80] + 256*KCBinFile[$0360-aadr+$80] -aadr+$80;
  ConvertBasic(strings, vMUEHLHAUSEN, start);
end;

procedure TKCFile.ConvertASCIIBasic(strings: TStrings);
var
  i: integer;
  s: String;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopf berspringen ('D3D3D3' + 8 Byte Name + 2 Byte Gesamtlnge)
  i := 13;

  while i<Length(KCBinFile) do begin
     //00-Bytes bergehen, Ende ist 03
     if KCBinFile[i] = 3 then break;
     if KCBinFile[i] <> 0 then
       s := s + chr(KCBinFile[i]);

     inc(i);
  end;

  strings.Text := s;

  Strings.EndUpdate;
end;

procedure TKCFile.ConvertText(strings: TStrings; offset: integer = 128);
var
  i: integer;
  s: String;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopfblock berspringen
  i := offset;

  while i<Length(KCBinFile) do begin
     //00-Bytes bergehen, Ende ist 03
     if not IgnoreBASICEnd then
       if KCBinFile[i] = 3 then break;
     if KCBinFile[i] <> 0 then
       s := s + chr(KCBinFile[i]);

     inc(i);
  end;

//  strings.Text := s;
  Strings.Text := AdjustLineBreaks(s);

  Strings.EndUpdate;
end;

procedure TKCFile.ConvertBasicode(strings: TStrings);
var
  i: integer;
  s: String;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopfblock berspringen
  i := 1;

  while i<Length(KCBinFile) do begin
     //Ende ist 0
     if KCBinFile[i] = 0 then break;
     s := s + chr(KCBinFile[i]);
     inc(i);
  end;
  strings.Text := s;

  Strings.EndUpdate;
end;

procedure TKCFile.ConvertForth(strings: TStrings);
var
  i: integer;
  s: String;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopfblock berspringen
  i := 128;

  while i<Length(KCBinFile) do begin
     //Ende ist 0
     if KCBinFile[i] = 0 then break;
     s := copy (pchar(@KCBinFile[i]),1,32);
     inc(i,32);
    strings.Add(s);
  end;

  Strings.EndUpdate;
end;

procedure TKCFile.ConvertText1(strings: TStrings);
var
  i: integer;
  s: string;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopf berspringen
  i := 128;

  while i<Length(KCBinFile) do begin
    //Zeile bearbeiten
    s := '';
    while ( KCBinFile[i] <> 0 )
      and ( i<=Length(KCBinFile) )
    do begin
        s := s + chr(KCBinFile[i]);
        inc(i);
    end;
    // Zeilenendezeichen (0)
    inc(i);

    //strings.Add(Format('%.4x %d %s', [addrnl, znr, s]));
    strings.Add(s);

    // Ende??
    //if KCBinFile[i] = 0 then break;
  end;

  Strings.EndUpdate;
end;

procedure TKCFile.ConvertScript(strings: TStrings);
var
  i, i0: integer;
  s: string;
  c: string;
  linelength: integer;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopf berspringen
  i := 256;

  linelength := KCBinFile[128+3]+1; // Block 1, 4. Byte

  while i<Length(KCBinFile) do begin
    //Zeile bearbeiten
    s := '';
    i0 := i;
    while ( KCBinFile[i] <> $0d )
      and ( i<=Length(KCBinFile) )
      and ( i-i0 < linelength)
    do begin
    	c := chr(KCBinFile[i]);
    	case chr(KCBinFile[i]) of
        #$00:	break;        // EOT
        // direktet Umwandlung der Umaute geht nicht wg. OEM-Z1013-Zeichensatz
        #$01:	c := 'AE';    // 
    	  #$02: c := 'OE';    // 
    	  #$03: c := 'UE';    // 
    	  #$04: c := 'ae';    // 
    	  #$05: c := 'oe';    // 
    	  #$06: c := 'ue';    // 
    	  #$07: c := 'ss';    // 
     //   #$08:
     //   #$09:
     //   #$0A:
     //   #$0B:
     //   #$0C:
     //   #$0D:	// neue Zeile
     //   #$0E:	//
     //   #$0F:	//
        #$10:	c := '';      // breit ein
        #$11:	c := '';      // breit aus
        #$12:	c := '';      // fett ein
        #$13:	c := '';      // fett aus
        #$14:	c := '';      // FF
     //   #$15:	// Variablen
        #$16:	c := '-';      // Softtrennstrich
     //   #$17:	// Schriftarten
     //   #$18:	//
     //   #$19:	//
     //   #$1A:	//
      end;
      s := s + c;
      inc(i);
    end;
    // Zeilenendezeichen (0D) berlesen
    if KCBinFile[i] = $0d then
      inc(i);

    strings.Add(s);

    if c = #0 then break;
  end;

  Strings.EndUpdate;
end;


(* Basic-gepackte Zahlen konvertieren.
Das funktioniert, leifert aber wg. grerer Genauigkeit nicht dieselben
Zahlen wie BASIC selbst *)
function   BasicNumber(b1,b2,b3,b4: byte): single;
var
  mantisse: integer;
  exp: integer;
begin
  //exponent
  exp := b4 and 127;
  if (b4 and 128) = 0 then exp := exp-128;
  //mantisse
  mantisse := (b3 and 127 + 128)*$100*$100 + b2*$100 + b4;

  if (b3 and 128) = 0 then
    result := mantisse * IntPower(2,exp-24)
  else
    result := - mantisse * IntPower(2,exp-24);
end;

// 18.02.2011
procedure TKCFile.ConvertBasicArray(strings: TStrings);
var
  i: integer;
  gesamtlaenge: integer;
  indx: integer;
  sa: array[0..8000] of string;
  k, p: integer;
  len: integer;
  bn: single;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopf berspringen ('D3D3D3' + 8 Byte Name + 2 Byte Gesamtlnge)
  i := 13;

  indx := -1; gesamtlaenge := 0;
  while i<Length(KCBinFile) do begin
    if (indx >= 0) and (gesamtlaenge = KCBinFile[i+1]*256+KCBinFile[i])
    then begin
      inc(i,2);
      strings.add(Format('DIM A$(%d)', [indx]));
      //weiter mit Auswertung
      for k := indx downto 0 do begin
        len := ord(sa[k][1]);
//        sa[k] := 'abc';     // substr(KCBinFile[i]..KCBinFile[i+len]
          sa[k]:= '';
          for p := i to i+len-1 do
            sa[k] := sa[k] + chr(KCBinFile[p]);
        inc(i, len);
      end;
      for k := 0 to indx do begin
        strings.Add(Format('A$(%d) = "%s"', [k, sa[k]]))
      end;
      break;
    end else begin
      if KCBinFile[i+1] <> ((KCBinFile[i] +1) and 255)
      then begin
    	  //kein Text-Array
    	  strings.add('Zahlen-Array (test)');
        i := 13; k := 0;
        while i<Length(KCBinFile) do begin
          if (KCBinFile[i] = 3) and (Length(KCBinFile) -i < 256) then break;
          bn := BasicNumber(KCBinFile[i], KCBinFile[i+1], KCBinFile[i+2], KCBinFile[i+3]);
          //strings.Add(Format('A(%d) = %2x %2x %2x %2x = %.6g', [k, KCBinFile[i], KCBinFile[i+1], KCBinFile[i+2], KCBinFile[i+3], bn]));
          strings.Add(Format('A(%d) = %.6g', [k, bn]));
          inc(i,4); inc(k,1);
        end;
  	    break;
      end else begin
        inc(indx);
        sa[indx] := chr(KCBinFile[i]);
        gesamtlaenge := gesamtlaenge + KCBinFile[i];
        inc(i,4);
      end;
    end;
  end;

  Strings.EndUpdate;
end;





procedure TKCFile.ConvertTextAll(strings: TStrings);
var
  i: integer;
  s: string;
begin
  strings.BeginUpdate;
  strings.Clear;

  // Kopf berspringen
  i := 128;

  SetLength(s, Length(KCBinFile)-i);
  move(KCBinFile[i], PChar(s)^, Length(KCBinFile)-i);

  for i:=1 to length(s) do
    if s[i] = #0 then s[i] := '.';

  Strings.Text := AdjustLineBreaks(s);

  Strings.EndUpdate;
end;


procedure TKCFile.ConvertHex(strings: TStrings);
var
  i, j, k, adr, blocknr: integer;
  s: string;
  firstblock: boolean;
begin
  strings.BeginUpdate;
  strings.Clear;
  i := 0;
  adr := 0;
  firstblock := true;
	
  while i<Length(KCBinaryFile) do begin
    blocknr := KCBinaryFile[i];
    if (blocknr = 0) and VirtualHeader then
      strings.Add(Format('Block %.2x (auto generated)', [blocknr]))
    else
      strings.Add(Format('Block %.2x', [blocknr]));
    i := i + 1;
    for k := 0 to 7 do begin
      setlength(s,16);
      move(KCBinaryFile[i],s[1],16);
      for j:=1 to length(s) do
        if s[j] < ' ' then s[j] := '.';

      strings.add(Format('%.4x: %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x  %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x  %s',
      [adr+k*16,
          KCBinaryFile[i+0],
          KCBinaryFile[i+1],
          KCBinaryFile[i+2],
          KCBinaryFile[i+3],
          KCBinaryFile[i+4],
          KCBinaryFile[i+5],
          KCBinaryFile[i+6],
          KCBinaryFile[i+7],
          KCBinaryFile[i+8],
          KCBinaryFile[i+9],
          KCBinaryFile[i+10],
          KCBinaryFile[i+11],
          KCBinaryFile[i+12],
          KCBinaryFile[i+13],
          KCBinaryFile[i+14],
          KCBinaryFile[i+15],
          s
      ]));
      i := i + 16;
    end;
    strings.Add('');
    if (firstblock) {and (KCData in [UNKNOWN, BASIC_KCB])} then begin
      adr := aadr;
      firstblock := false;  
    end else
      adr := adr + 128;
  end;
  Strings.EndUpdate;
end;

procedure TKCFile.GetKCData;
begin
  if (filetyp = 'BASIC') and (pos(fileext, 'SSSWWW') mod 3 = 1) then
    KCData := BASIC_CSAVE
  else if (filetyp = 'BASIC') and  (pos(fileext, 'TTTXXX') mod 3 = 1 ) then
    KCData := BASIC_ARRAY
  else if (filetyp = 'BASIC') and  (pos(fileext, 'UUUYYY') mod 3 = 1 ) then
    KCData := BASIC_LIST
  else if (filetyp = 'MC') and (fileext = 'ASM') then
   KCData := ASM_EDAS //(KC85/2)
  else if (filetyp = 'MC') and (fileext = 'BAC') then
    KCData := BASICODE
  else if (filetyp = 'MC') and (pos(fileext, '(F)FORF83') mod 3 = 1) then
    KCData := FORTH
  else if (filetyp = 'MC') and (fileext = 'TXT') and ((aadr = $2F3F) or (aadr = $3014)) then
    KCData := TEXT1
  else if (filetyp = 'MC') and (fileext = 'TXT') then
    KCData := TEXT_ASM
  else if (filetyp = 'MC') and (fileext = 'TX2') then
    KCData := SCRIPT
  else if (filetyp = 'MC') and (KCData = BASIC_KCB) then
    KCData := BASIC_KCB
  else
    KCData := UNKNOWN
  ;
end;

procedure TKCFile.DecodeKCFile(strings: TStrings; typ: TKCData = UNDEF);
begin
  strings.Clear;

  if typ = UNDEF then
     typ := KCData;
  case typ of
    BASIC_CSAVE:  begin
                    if Vendor <> vMUEHLHAUSEN then 
                       ConvertBasic(strings, Vendor); // 1st run vendor detection
                    ConvertBasic(strings, Vendor); // 2nd run convert
                  end;
    BASIC_ARRAY:  ConvertBasicArray(strings);
    BASIC_LIST:   ConvertASCIIBasic(strings);
    ASM_EDAS:     ConvertText(strings);
    BASICODE:     ConvertBasicode(strings);
    FORTH:        ConvertForth(strings);
    TEXT1:        ConvertText1(strings);
    TEXT_ASM:     ConvertText(strings);
    SCRIPT:       ConvertScript(strings);
    UNKNOWN:      ConvertHex(strings);
    BASIC_KCB:    ConvertBasicKCB(strings);
    else          ConvertTextAll(strings);
  end;
end;

end.

