{
  WLX TAP Lister Plugin
  (c) 2006-2008 Volker Pohlers
  based on
  WLX Delphi Tutorial Plugin
  (c) 2004 Mutex Ltd. mutex@nm.ru
}

{ History:
	12.06.2007  Vorbereitung fr Multipart-AF-Tapes
	13.06.2007  HEX-Viewer, autom. Laden des Fonts
  14.06.2007  fr TC7: ListLoadNext-Untersttzung
  17.09.2008  fr KC-M: neue Formate KCC und KCB, Dank an M.Leubner

  30.10.2018  bei robotron BASIC-Dateien als .ZBS (mit 11 Byte Kopf) statt .SSS (ohne 11 Byte Kopf) speichern
}

{ todo: Anzeige der Parameter beim KC-M (Anzahl + Werte)
        Speichern der Darstellungsoption Mhhausen/robotron
        Mhlhausen-Zeichensatz einbinden
}


unit unMain;

interface

uses
  Windows, Messages, Controls, StdCtrls, ComCtrls, Classes, Menus, Forms,
  AppEvnts, SysUtils, uKCFile, ExtCtrls, Dialogs, ImgList;

type
  TfmMain = class(TForm)
    PopupMenu1: TPopupMenu;
    Help1: TMenuItem;
    About1: TMenuItem;
    StringViewer: TMemo;
    mrZ9001: TMenuItem;
    mrKC85241: TMenuItem;
    N2: TMenuItem;
    Wordwrap1: TMenuItem;
    N1: TMenuItem;
    StatusBar: TStatusBar;
    Export1: TMenuItem;
    SaveDialog: TSaveDialog;
    IgnoreBASICEnd1: TMenuItem;
    Splitter1: TSplitter;
    ImageList1: TImageList;
    ListView1: TListView;
    HexView1: TMenuItem;
    ForceBasic1: TMenuItem;
    N3: TMenuItem;
    Charset1: TMenuItem;
    procedure StringViewerKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure Help1Click(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure mrZ9001Click(Sender: TObject);
    procedure Wordwrap1Click(Sender: TObject);
    procedure Export1Click(Sender: TObject);
    procedure IgnoreBASICEnd1Click(Sender: TObject);
    procedure ForceViewClick(Sender: TObject);
    procedure Charset1Click(Sender: TObject);
  private
    TotCmdWin: HWND;    //handle of TC window
    ParentWin: HWND;    //handle of Lister window
    QuickView: boolean; //Ctrl+Q panel
    viewfilename: String;
    KCFile: TKCFile;
    procedure AppException(Sender: TObject; E: Exception);
    procedure LoadTAP(const FileToView: string);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
  public
    //our constructor
    constructor CreateParented(ParentWindow: HWND; const FileToView: string); reintroduce;
  end;

function ShowTAP(ListerWin: HWND; FileToLoad: string): HWND;
function ShowNextTAP(PluginWin: HWND; FileToLoad: string): integer;
procedure HideTAP(PluginWin: HWND);

implementation

uses unAbout;

{$R *.dfm}

procedure wMsgBox(hWnd: HWND; Msg: string);
begin
  MessageBox(hWnd, PChar(Msg), 'Message', MB_OK+MB_ICONINFORMATION);
end;

procedure TfmMain.StringViewerKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  //Alt+X -> TC close:
  if (Shift = [ssAlt]) and (Chr(Lo(Key)) = 'X') then begin
    Application.Handle := 0; //C++ GPF (Delphi AV)
    Application.RemoveComponent(Self);
    PostMessage(TotCmdWin, WM_SYSCOMMAND, SC_CLOSE, 0);
    exit;
  end;
  //hot keys of Lister:
  if Shift = [] then begin
    if Key = VK_ESCAPE then begin                 //File -> Exit (ESC)
      if not QuickView then
        PostMessage(ParentWin, WM_KEYDOWN, VK_ESCAPE, 0)
      else
        PostMessage(ParentWin, WM_KEYDOWN, VK_TAB, 0);
      Key := 0;
    end
    else if Chr(Lo(Key)) in ['N', 'P'] then begin
      PostMessage(ParentWin, WM_KEYDOWN, Key, 0); //File -> Next (N) or Prev (P)
      Key := 0;
    end
    else if Chr(Lo(Key)) in ['1'..'7'] then begin
      PostMessage(ParentWin, WM_KEYDOWN, Key, 0); //Options -> 1..7
      Key := 0;
    end
  end;
  //own dialogs:
  if (Shift = []) and (Key = VK_F1) then  //F1
    Help1.Click;
  if (Shift = []) and (Key = VK_F2) then  //F2
    About1.Click;
  if (Shift = []) and (Key = VK_F8) then  //F8
    Export1.Click;
  if (Shift = []) and (Key = Ord('W')) then begin
    Wordwrap1.Checked := not Wordwrap1.Checked;
    Wordwrap1Click(Self);
  end;
  if (Shift = []) and (Key = Ord('H')) then begin
    HexView1.Checked := not HexView1.Checked;
    ForceBasic1.Checked := false;
    ForceViewClick(Self);
  end;
  if (Shift = []) and (Key = Ord('F')) then begin
    ForceBasic1.Checked := not ForceBasic1.Checked;
    HexView1.Checked := false;
    ForceViewClick(Self);
  end;
  if (Shift = []) and (Key = Ord('C')) then begin
    Charset1Click(self);
    ForceViewClick(Self);
  end;
  if (Shift = []) and (Key = Ord('I')) then begin
    IgnoreBASICEnd1.Checked := not IgnoreBASICEnd1.Checked;
    IgnoreBASICEnd1Click(Self);
  end;
  if (Shift = []) and (Key = Ord('B'))
  and (KCFile.filetyp = 'BASIC')
  then begin
    //if KCFileHeader.Vendor = vROBOTRON then KCFileHeader.Vendor := vMUEHLHAUSEN else KCFileHeader.Vendor := vROBOTRON;
    //mrZ9001.Checked := ( KCFileHeader.Vendor = vROBOTRON );
    //mrKC85241.Checked := ( KCFileHeader.Vendor = vMUEHLHAUSEN );
    mrZ9001.Checked := not mrZ9001.Checked ;
    mrZ9001Click(Self);
  end;
  StringViewer.SetFocus;
end;

// von http://www.martinstoeckli.ch/delphi/delphi.html#AppVersion
function Sto_GetFmtFileVersion(const FileName: String;
  const Fmt: String = '%d.%d.%d.%d'): String;
var
  iBufferSize: DWORD;
  iDummy: DWORD;
  pBuffer: Pointer;
  pFileInfo: Pointer;
  iVer: Array[1..4] of Word;
begin
  // set default value
  Result := '';
  // get size of version info (0 if no version info exists)
  iBufferSize := GetFileVersionInfoSize(PChar(FileName), iDummy);
  if (iBufferSize > 0) then
  begin
    GetMem(pBuffer, iBufferSize);
    try
    // get fixed file info
    GetFileVersionInfo(PChar(FileName), 0, iBufferSize, pBuffer);
    VerQueryValue(pBuffer, '\', pFileInfo, iDummy);
    // read version blocks
    iVer[1] := HiWord(PVSFixedFileInfo(pFileInfo)^.dwFileVersionMS);
    iVer[2] := LoWord(PVSFixedFileInfo(pFileInfo)^.dwFileVersionMS);
    iVer[3] := HiWord(PVSFixedFileInfo(pFileInfo)^.dwFileVersionLS);
    iVer[4] := LoWord(PVSFixedFileInfo(pFileInfo)^.dwFileVersionLS);
    finally
      FreeMem(pBuffer);
    end;
    // format result string
    Result := Format(Fmt, [iVer[1], iVer[2], iVer[3], iVer[4]]);
  end;
end;

function Sto_GetModuleName: String;
var
  szFileName: array[0..MAX_PATH] of Char;
begin
  GetModuleFileName(hInstance, szFileName, MAX_PATH);
  Result := szFileName;
end;


procedure TfmMain.Help1Click(Sender: TObject);
begin
  wMsgBox(Handle, 'TAP file Lister plugin v '+Sto_GetFmtFileVersion(Sto_GetModuleName)+#10
    +#10'B - switch Basic (robotron/Mhlhausen)'
    +#10'I - ignore Basic end'
    +#10'H - toggle hex view'
    +#10'F - force Basic view'
    +#10'W - toggle wordwrap'
    +#10'F1 - this help'
    +#10'F2 - About'
    +#10'F8 - export as KCC/SSS/TAP (overwrites existing files without warning!)'
  );
  StringViewer.SetFocus;
end;

procedure TfmMain.About1Click(Sender: TObject);
var fmAbout: TfmAbout;
begin
  fmAbout := TfmAbout.Create(Self);
  try
    fmAbout.Label1.Caption := 'Version '+Sto_GetFmtFileVersion(Sto_GetModuleName);
    fmAbout.ShowModal;
  finally
    fmAbout.Free;
  end;
  StringViewer.SetFocus;
end;

procedure TfmMain.AppException(Sender: TObject; E: Exception);
begin
  wMsgBox(Handle, 'Plugin Error:' + #13 + E.Message);
end;

procedure TfmMain.Charset1Click(Sender: TObject);
begin
  with StringViewer.Font do begin
    if name = 'Z1013 Text' then name := 'KC85/4 CAOS Font Text'
    else if name = 'KC85/4 CAOS Font Text' then name := 'Z1013 Umlaut'
    else name := 'Z1013 Text';
  end;
end;

procedure TfmMain.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := (WS_CHILD or WS_MAXIMIZE) and not WS_CAPTION and not WS_BORDER;
  Params.WindowClass.cbWndExtra := SizeOf(Pointer); //4 bytes for address of form
end;

constructor TfmMain.CreateParented(ParentWindow: HWND; const FileToView: string);
const WinCmdClassName = 'TTOTAL_CMD';
begin //non standard constructor
  inherited CreateParented(ParentWindow); //our window is child window
  TotCmdWin := FindWindow(WinCmdClassName, nil);
  ParentWin := ParentWindow;
  QuickView := GetParent(ParentWin) <> 0;

  KCFile := TKCFile.Create;
  LoadTAP(FileToView);
end;
{end of form}


procedure TfmMain.LoadTAP(const FileToView: string);
var
  typ: TKCFileType;
begin
    try
      viewfilename := FileToView;
      KCFile.LoadFile(FileToView, typ);
      KCFile.DecodeKCFile(StringViewer.Lines);
      case KCFile.KCData of
        BASIC_CSAVE: Statusbar.Panels[0].Text := 'BASIC CSAVE';
        BASIC_ARRAY: Statusbar.Panels[0].Text := 'BASIC CSAVE*';
        BASIC_LIST:  Statusbar.Panels[0].Text := 'BASIC LIST';
        ASM_EDAS:    Statusbar.Panels[0].Text := 'ASM EDAS (KC85/2)';
        FORTH:       Statusbar.Panels[0].Text := 'Forth';
        TEXT1:       Statusbar.Panels[0].Text := 'TEXT1';
        SCRIPT:      Statusbar.Panels[0].Text := 'SCRIPT';
        TEXT_ASM:    Statusbar.Panels[0].Text := 'Text/ASM';
        BASICODE:    Statusbar.Panels[0].Text := 'BASICODE';
        BASIC_KCB:   Statusbar.Panels[0].Text := 'BASIC (KCB)';
        else {UNKNOWN:} Statusbar.Panels[0].Text := '(unknown)';
      end;
      if KCFile.filetyp = 'BASIC' then begin
        mrKC85241.Checked := ( KCFile.Vendor = vMUEHLHAUSEN );
        if KCFile.Vendor = vROBOTRON then begin
          Statusbar.Panels[0].Text := Statusbar.Panels[0].Text + ' ROBOTRON';
          StringViewer.Font.Name := 'Z1013 Text';
        end else begin
          Statusbar.Panels[0].Text := Statusbar.Panels[0].Text + ' Mhlhausen';
          StringViewer.Font.Name := 'KC85/4 CAOS Font Text';
        end;
      end;
      with KCFile do
        Statusbar.Panels[1].Text := Format('(%s) %s.%s address %.4x-%.4x, start %.4x, first block %d'
          ,[filetyp, filename, fileext, aadr, eadr, sadr, first_blocknr]);
    except
    end;
end;

procedure TfmMain.ForceViewClick(Sender: TObject);
begin
  if HexView1.Checked then
    KCFile.DecodeKCFile(StringViewer.Lines, UNKNOWN)
  else if ForceBasic1.Checked then begin
    KCFile.IgnoreBASICEnd := true;
    KCFile.DecodeKCFile(StringViewer.Lines, BASIC_CSAVE);
  end else
    KCFile.DecodeKCFile(StringViewer.Lines);
end;

//----------------------------------------------------------------------------------
{initialization and finalization}
type
  TPlugInfo = record
    PlugWinProc: Pointer; //callback function of our form
    PlugForm: TfmMain;    //our form
  end;

function HookDestroy(PluginWin: HWND; Msg, wParam, lParam: LongInt): LongInt; stdcall;
var p: ^TPlugInfo;
begin //hook destroy our window
  p := Pointer(GetWindowLong(PluginWin, GWL_USERDATA));
  if Msg <> WM_DESTROY then
    Result := CallWindowProc(p^.PlugWinProc, PluginWin, Msg, wParam, lParam)
  else begin //plugin close
    HideTAP(PluginWin);
    Result := 0;
  end;
end;

procedure HideTAP(PluginWin: HWND);
var p: ^TPlugInfo;
begin //finalization
  p := Pointer(GetWindowLong(PluginWin, GWL_USERDATA));
  with p^.PlugForm do begin
    try
      Application.RemoveComponent(p^.PlugForm);
      Application.Handle := 0;
      //restore callback function
      SetWindowLong(Handle, GWL_WNDPROC, Integer(p^.PlugWinProc));
      Free;
    except
      on E: Exception do
        wMsgBox(0, 'DestroyWindow error:' + #13 + E.Message);
    end;
  end;
  Dispose(p);
end;

function ShowTAP(ListerWin: HWND; FileToLoad: string): HWND;
var fmMain: TfmMain; s: string; p: ^TPlugInfo;
begin //initialization
  try
    s := ExtractFilePath(FileToLoad);
    if not SetCurrentDir(s) then  //folder of RTF
      raise Exception.Create('Error of SetCurrentDir() for Folder: ' + s);
    fmMain := TfmMain.CreateParented(ListerWin, FileToLoad);
    fmMain.Show;
    //synchronize our form and Lister
    Application.Handle := ListerWin;
    Application.OnException := fmMain.AppException;
    Application.InsertComponent(fmMain);
    //substitution callback function
    New(p);
    SetWindowLong(fmMain.Handle, GWL_USERDATA, Integer(p));
    p^.PlugForm := fmMain;
    p^.PlugWinProc := Pointer(SetWindowLong(fmMain.Handle, GWL_WNDPROC, Integer(@HookDestroy)));
    //set focus to our window
    if not fmMain.QuickView then begin
      PostMessage(fmMain.Handle, WM_SETFOCUS, 0, 0);
      fmMain.StringViewer.SetFocus;
    end;
    Result := fmMain.Handle;
  except
    on E: Exception do begin
      wMsgBox(ListerWin, 'Open error:' + #13 + E.Message);
      Result := 0;
    end;
  end;
end;

function ShowNextTAP(PluginWin: HWND; FileToLoad: string): integer;
var p: ^TPlugInfo;
begin //finalization
  p := Pointer(GetWindowLong(PluginWin, GWL_USERDATA));
  with p^.PlugForm do begin
  try
//    Show;
//    KCFile := TKCFile.Create;
    LoadTAP(FileToLoad);
    Result := 0;
  except
    on E: Exception do begin
      wMsgBox(PluginWin, 'Open error:' + #13 + E.Message);
      Result := -1;
    end;
  end;
  end;
end;


procedure TfmMain.mrZ9001Click(Sender: TObject);
begin
  if mrZ9001.Checked then KCFile.Vendor := vROBOTRON else KCFile.Vendor := vMUEHLHAUSEN;
    mrZ9001.Checked := ( KCFile.Vendor = vROBOTRON );
    mrKC85241.Checked := ( KCFile.Vendor = vMUEHLHAUSEN );

  if KCFile.filetyp = 'BASIC' then  begin
    KCFile.DecodeKCFile(StringViewer.Lines);
//    mrZ9001.Checked := ( KCFileHeader.Vendor = vROBOTRON );
//    mrKC85241.Checked := ( KCFileHeader.Vendor = vMUEHLHAUSEN );
  end;

  if mrKC85241.Checked then
    StringViewer.Font.Name := 'KC85/4 CAOS Font Text'
  else
    StringViewer.Font.Name := 'Z1013 Text';
end;

procedure TfmMain.Wordwrap1Click(Sender: TObject);
begin
  StringViewer.WordWrap := Wordwrap1.Checked;
  if  StringViewer.WordWrap then
    StringViewer.ScrollBars := ssVertical
  else
    StringViewer.ScrollBars := ssBoth;
end;

procedure TfmMain.Export1Click(Sender: TObject);
var
  typ: TKCFileType;
begin
  with SaveDialog do begin
    if UpperCase(ExtractFileExt(viewfilename)) = '.TAP' then begin
      if KCFile.filetyp = 'BASIC' then begin
      	 if KCFile.Vendor = vMUEHLHAUSEN then
           filename := KCFile.filename + '.' + KCFile.fileext
         else
           filename := KCFile.filename + '.ZBS';
         typ := ftBASIC;
      end else begin
         filename := KCFile.filename + '.KCC';
         typ := ftOHNE_BLOCKNR;
      end;
    end else begin
      filename := KCFile.filename + '.tap';
      typ := ftAF_TAPE;
    end;

    //if Execute then
      KCFile.SaveFile(filename, typ);
  end;
end;

procedure TfmMain.IgnoreBASICEnd1Click(Sender: TObject);
begin
  KCFile.IgnoreBASICEnd := IgnoreBASICEnd1.Checked;
//  if KCFile.filetyp = 'BASIC' then  begin
    KCFile.DecodeKCFile(StringViewer.Lines);
//  end;
end;

var
  fontname: string;

initialization
  fontname := ExtractFilePath(Sto_GetModuleName)+'hcddr.fon';
  AddFontResource(PChar(fontname));
  SendMessage(HWND_BROADCAST, WM_FONTCHANGE, 0, 0);

finalization
  RemoveFontResource(PChar(fontname));
  SendMessage(HWND_BROADCAST, WM_FONTCHANGE, 0, 0);
end.

