#
# linter.py
# Linter for SynLint
#
# Original: by Aparajita Fishman
# Copyright (c) 2013 Aparajita Fishman
# Change for SynLint: Alexey(Synwrite)
#
# License: MIT
#

"""This module exports the PHPLint plugin class."""

from SublimeLinter.lint import Linter


class PHPLint(Linter):
    """Provides an interface to the phplint executable."""

    syntax = ('PHP', 'PHP_')

    cmd = 'phplint --print-path shortest --print-context --tab-size 4 --no-overall'
    version_args = '--version'
    version_re = r'PHPLint (?P<version>\d+\.\d+)'
    version_requirement = '>= 1.1'
    regex = (
        r'(?i)^(?:'
        r'\t.*?\r?\n'
        r'\t(?P<col>.*) \\_ HERE\r?\n)?'
        r'(?P<filename>.+?):(?P<line>\d+): '
        r'(?:(?P<error>(?:fatal )?error)|(?P<warning>warning|notice)): '
        r'(?P<message>[^`\r\n]*(?:`(?P<near>[^\']+)\')?[^\r\n]*)'
    )
    multiline = True
    tempfile_suffix = 'php'

    def split_match(self, match):
        """Return the match with ` quotes transformed to '."""
        match, line, col, error, warning, message, near = super().split_match(match)

        if message == 'no PHP code found at all':
            match = None
        else:
            message = message.replace('`', '\'')

            # If the message contains a complaint about a function
            # and near looks like a function reference, remove the trailing
            # () so it can be found.
            if 'function \'' in message and near and near.endswith('()'):
                near = near[:-2]

        return match, line, col, error, warning, message, near
