/* 8/16/2004
 *
 * FileForum.ru:
 *  http://www.fforum.ru/index.php?s=390d85f944c93d3690cf437b3c3d0b8a&act=ST&f=26&t=4817&unread=&st=0&
 *  http://www.fforum.ru/viewtopic.php?f=26&t=4817
 *
 * You can send TC the following command:
 *
 *     wm_InvokeMenuCommand=WM_USER+51
 *
 * with WPARAM set to the command you want to pass to TC.
 *
 * 26.06.2005 - added switch '-a' - send message to all TC windows
 * 17.11.2022 - added switch '-f' - bring TC window to foreground
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <windows.h>

#pragma comment(lib, "user32")

//#define ENUM_CHILDS
//#define DEBUG
#ifdef DEBUG
#define DPRINTF fprintf
#else
#define DPRINTF while (0) fprintf
#endif /* DEBUG */

#define TCCP_VER    "0.3"
#define TCCP_DATE   "Nov 2022"

#define WM_USER_TC  WM_USER+51

#define PROG_LOGO \
"Total Commander Command Poster version " TCCP_VER ", " TCCP_DATE ".\n" \
"Post/send command(s) to TC window by number or name defined in Totalcmd.inc\n" \
"using Win32 API PostMessage()/SendMessage() with WM_USER+51 and wParam=cmd.\n\n"

#define PROG_USAGE \
"Usage:\n" \
"    tccp <cmd> [<cmd>...] [-i<tc_inc>] [-s] [-t<class_name>]\n" \
"\n" \
"<cmd> - numeric code (dec) or name of command,\n" \
"        e.g. 493 or cm_ConfigSavePos to save window position\n" \
"   -a - send commands to all found TC windows (default: first found window)\n" \
"   -f - bring TC window to foreground before sending command\n" \
"   -i - parse command(s) code from file <tc_inc>, e.g. -ic:\\wincmd\\TOTALCMD.INC\n" \
"        default: TOTALCMD.INC in working directory\n" \
"   -s - use SendMessage() function instead of PostMessage()\n" \
"   -t - TC main window class; in version 6.03 and default is \'TTOTAL_CMD\'\n" \
"   -? - help\n" \
"\n" \
"ERRORLEVEL: 0 - success or TC not started, non zero - some (m.b. unknown) error.\n" \
"\n" \
"Switches and command names are NOT case sensitive. It\'s order - arbitrary.\n" \
"Written by Hobo, mailto:hobo-mts@mail.ru\n" \
"See also: http://www.fforum.ru/viewtopic.php?f=26&t=4817\n"


static char TC_class[MAX_PATH] = "TTOTAL_CMD";  /* TC main window class */
static int  TC_wincnt = 0;          /* Count of TC windows */
static HWND *TC_winlist = NULL;     /* List of main TC winds. NULL - no more */


#if defined(DEBUG) & defined(ENUM_CHILDS)
BOOL CALLBACK
enum_TC_childs(HWND hwnd, LPARAM lParam)
{
    char    buf[MAX_PATH];
    GetWindowText(hwnd, buf, MAX_PATH);
    CharToOem(buf, buf);
    if (strlen(buf)) DPRINTF(stderr, "\t'%s'\n", buf);
    return 1;
}
#endif


/* See EnumWindowsProc() from Windows */
BOOL CALLBACK
enum_TC_windows(HWND hwnd, LPARAM lParam)
{
    char    buf[MAX_PATH];

    if (!GetClassName(hwnd, buf, MAX_PATH)) return 1;
    if (_stricmp(buf, TC_class)) return 1;
    if (TC_winlist) {
        TC_winlist[TC_wincnt] = hwnd;
#ifdef DEBUG
        GetWindowText(TC_winlist[TC_wincnt], buf, MAX_PATH);
        fprintf(stderr, "%d: '%s'\n", TC_wincnt, buf);
#endif
#if defined(DEBUG) & defined(ENUM_CHILDS)
        EnumChildWindows(hwnd, enum_TC_childs, 0);
#endif
    }
    ++TC_wincnt;
    return 1;
}


/* Makes list of all opened TC windows
 * Returns number of windows or <0 on error */
static int
make_TC_winlist(void)
{
    TC_wincnt = 0;
    if (!EnumWindows(enum_TC_windows, 0)) {
        fprintf(stderr, "EnumWindows() error (%d)\n", __LINE__);
        return -1;
    }
    DPRINTF(stderr, "TC windows count: %d\n", TC_wincnt);
    if (!(TC_winlist = calloc(TC_wincnt+1, sizeof(HWND)))) {
        fprintf(stderr, "calloc() error (%d)\n", __LINE__);
        return -2;
    }
    TC_wincnt = 0;
    if (!EnumWindows(enum_TC_windows, 0)) {
        fprintf(stderr, "EnumWindows() error (%d)\n", __LINE__);
        return -3;
    }
    return TC_wincnt;
}


extern int
main(int argc, char *argv[])
{
    int     to_all = 0;             /* to all TC windows */
    int     to_fg = 0;              /* bring to foreground */
    int     *TC_commands = NULL;    /* Commands to send/post */
    char    inc_path[MAX_PATH] = "TOTALCMD.INC";    /* File with command codes names */
    int     use_Send = 0;           /* !0 - use SendMessage(), 0 - PostMessage() */
    HWND    tcwnd = NULL;
    char    *s, tcwn[MAX_PATH+1];
    int     cmdc = 0, rc = 0;
    int     i, j;

    /* parse switches and count commands */
    for (i = 1, cmdc = 0; i < argc; ++i) {
#ifdef DEBUG
        CharToOem(argv[i], tcwn);
        DPRINTF(stderr, "%d: '%s'\n", i, tcwn);
#endif
        s = argv[i];
        if (*s != '-') {
            ++cmdc;
            continue;
        }
        ++s;
        switch (tolower(*s)) {
        case 'a':
            to_all = 1;
            break;
        case 'f':
            to_fg = 1;
            break;
        case 'i':
            strcpy(inc_path, s+1);
            break;
        case 's':
            use_Send = 1;
            break;
        case 't':
            strcpy(TC_class, s+1);
            break;
        case '?':
            printf(PROG_LOGO PROG_USAGE);
            rc = 0;
            goto stop;
            break;
        default:
            fprintf(stderr, "Invalid switch -%s. Use tccp -? for help.\n", s);
            rc = 1;
            goto stop;
        }
    }
    if (!cmdc) {
        printf(PROG_LOGO PROG_USAGE);
        rc = 1;
        goto stop;
    }
    if (!(TC_commands = malloc(cmdc * sizeof(int *)))) {
        fprintf(stderr, "malloc() error (%d)\n", __LINE__);
        rc = 2;
        goto stop;
    }

    /* parse command, number or name */
    for (i = 1, j = 0; i < argc; ++i) {
        s = argv[i];
        if (*s == '-') continue;    /* skip switch */
        if (isdigit(*s)) {          /* numeric */
            TC_commands[j] = atoi(s);
        } else {                    /* symbolic */
            FILE    *fp;
            char    buf[1024], *s1, *ch;
            int     found = 0;

            if (!(fp = fopen(inc_path, "r"))) {
                fprintf(stderr, "Can not open '%s'\n", inc_path);
                rc = 1;
                goto stop;
            }
            while (fgets(buf, sizeof(buf), fp) && !found) {
                if (ch = strchr(buf, '=')) {
                    for (s1 = buf; *s && *s <= ' '; ++s) ;
                    *ch++ = 0;
                    if (!_stricmp(s, s1)) {
                        found = 1;
                        TC_commands[j] = atoi(ch);
                    }
                }
            }
            fclose(fp);
            if (!found) {
                fprintf(stderr, "Command code not found for '%s'.\n", s);
                rc = 1;
                goto stop;
            }
        }
        ++j;
    }
#ifdef DEBUG
    DPRINTF(stderr, "Commands: ");
    for (i = 0; i < cmdc; ++i) DPRINTF(stderr, " %d", TC_commands[i]);
    CharToOem(inc_path, tcwn);
    DPRINTF(stderr, "\nuse_Send:%d, inc_path: '%s', TC_class: '%s', to_all:%d.\n", use_Send, tcwn, TC_class, to_all);
#endif

    if (make_TC_winlist() < 0) {
        fprintf(stderr, "Error enumerating windows\n");
        rc = 3;
    }
    if (TC_wincnt == 0) {
        fprintf(stderr, "Total Commander main window not found.\n");
        rc = 0;
        goto stop;
    }
    if (!to_all) TC_wincnt = 1;

    for (j = 0; j < TC_wincnt; ++j) {
        tcwnd = TC_winlist[j];
        if (to_fg) {
            //SetWindowPos(tcwnd, HWND_TOP, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_SHOWWINDOW);
            SetForegroundWindow(tcwnd);
        }
        GetWindowText(tcwnd, tcwn, sizeof(tcwn));
        printf("%s: '%s': ", use_Send ? "S" : "P", tcwn); fflush(stdout);
        for (i = 0; i < cmdc; ++i) {
            if (use_Send) {
                printf("%u=>", TC_commands[i]); fflush(stdout);
                printf("%u ", SendMessage(tcwnd, WM_USER_TC, TC_commands[i], 0));
            } else {
                printf("%u->", TC_commands[i]); fflush(stdout);
                if (!PostMessage(tcwnd, WM_USER_TC, TC_commands[i], 0)) {
                    printf("Failed ");
                    rc = 3;
                    goto stop;
                } else {
                    printf("Ok ");
                }
            }
            fflush(stdout);
        }
        printf("\n");
        fflush(stdout);
    }

stop:
    if (TC_commands) { free(TC_commands); TC_commands = NULL; }
    if (TC_winlist) { free(TC_winlist); TC_winlist = NULL; }
    return rc;
}

// vim: set cc=81 :
