#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include "contentplug.h"

#pragma comment(lib, "user32")


#define WDXTEST_LOGO \
"Total Commander WDX plugin Test v0.2. Copyright (c) 2007 Oleg Bondar.\n" \
"Implements content plugin interface functionality.\n"

#define WDXTEST_USAGE \
"Usage:\n" \
"  wdxtest <wdx> <file> [-f[<N>]]\n" \
"\n" \
"<wdx>   - path to WDX plugin\n" \
"<file>  - path to file\n" \
"\n" \
"-f[<N>] - out ft_fulltext fields. <N> - maximumum number of bytes to output.\n" \
"          If <N> not speicified, out all field data\n" \
"\n" \
"If only <wdx> specified wdxtest shows plugin's supported fields.\n" \
"If <file> specified too it shows contents of all fields with all units.\n" \
"\n" \
"ERRORLEVEL: 0 - success, non zero - some (m.b. unknown) error.\n" \
"Program NOT tested with file names that contains non-ASCII chars.\n"

#define WTE_NO_WCX		1
#define WTE_NO_PROC		2
#define WTE_NO_FIELDS	3
#define WTE_NO_FILE		4
#define WTE_OPEN_FAIL	5
#define WTE_CLOSE_FAIL	6
#define WTE_NO_MEM		7
#define WTE_INV_OPT		8

#define SBUF_SIZE		1024


typedef struct WDX_FIELD {
	int		type;
	char    *name;
	char	*units;
} WDX_FIELD, *PWDX_FIELD;


static int		(__stdcall *pContentGetDetectString)(char *DetectString, int maxlen) = NULL;
static void		(__stdcall *pContentPluginUnloading)(void);
/* mandatory functions */
static int		(__stdcall *pContentGetSupportedField)(int FieldIndex, char *FieldName, char *Units, int maxlen) = NULL;
static int		(__stdcall *pContentGetValue)(char *FileName, int FieldIndex, int UnitIndex, void *FieldValue, int maxlen, int flags) = NULL;
static void		(__stdcall *pContentSetDefaultParams)(ContentDefaultParamStruct *dps) = NULL;

static int		out_fulltext = 0;
static int		max_fulltext = 0;


/*
 * Print formatted data to a stream in OEM code page.
 * Returns the number of bytes written
 */
static int
oprintf(FILE *fp, TCHAR *format, ...)
{
	va_list		marker;
	char		tbuf[4096];
	int			rc;

	va_start(marker, format);     /* Initialize variable arguments. */
	rc = vsprintf(tbuf, format, marker);
	CharToOem(tbuf, tbuf);
	fputs(tbuf, fp);
	return rc;
}


static char *
WDX_err_msg(int code)
{
	static char		buf[256];

	switch(code) {
		case ft_nosuchfield:	strcpy(buf, "ft_nosuchfield: Invalid field number given"); break;
		case ft_fileerror:		strcpy(buf, "ft_fileerror: File i/o error"); break;
		case ft_fieldempty:		strcpy(buf, "ft_fieldempty: Field valid, but empty"); break;
		case ft_ondemand:		strcpy(buf, "ft_ondemand: Field will be retrieved only when user presses <SPACEBAR>"); break;
		case ft_delayed:		strcpy(buf, "ft_delayed: Field takes a long time to extract -> try again in background"); break;
		default: 				sprintf(buf, "Unknown error code (%d)", code); break;
	}
	return buf;
}


static int
print_fval(int ft, char *buf)
{
	tdateformat		*df;
	ttimeformat		*tf;
	SYSTEMTIME		st;
	FILETIME		ftm;

	switch (ft) {
	case ft_numeric_32:
		oprintf(stdout, "%d\n", *(int*)buf);
		break;
	case ft_numeric_64:
		oprintf(stdout, "%ld\n", *(long int*)buf);
		break;
	case ft_numeric_floating:
		oprintf(stdout, "%g", *(double*)buf);
		if (*(buf+sizeof(double))) oprintf(stdout, " (%s)\n", buf+sizeof(double));
		else oprintf(stdout, "\n");
		break;
	case ft_date:
		df = (tdateformat*)buf;
		oprintf(stdout, "%04d/%02d/%02d\n", df->wYear, df->wMonth, df->wDay);
		break;
	case ft_time:
		tf = (ttimeformat*)buf;
		oprintf(stdout, "%02d:%02d:%02d\n", tf->wHour, tf->wMinute, tf->wSecond);
		break;
	case ft_boolean:
		if (*(int*)buf) oprintf(stdout, "TRUE\n");
		else oprintf(stdout, "FALSE\n");
		break;
	case ft_multiplechoice:
		oprintf(stdout, "%s\n", buf);
		break;
	case ft_string:
		oprintf(stdout, "'%s'\n", buf);
		break;
	case ft_fulltext:
		oprintf(stdout, "ft_fulltext unsupported\n");
		break;
	case ft_datetime:
		FileTimeToLocalFileTime((FILETIME*)buf, &ftm);
		FileTimeToSystemTime(&ftm, &st);
		oprintf(stdout, "%04d/%02d/%02d %02d:%02d:%02d\n",
			st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond);
		break;
	default:
		oprintf(stdout, "-- %s\n", WDX_err_msg(ft));
		break;
	}
	return 0;
}


extern int
main(int argc, char *argv[])
{
	HINSTANCE	hwdx = NULL;
	WDX_FIELD	*wf = NULL;
	int			wfcnt = 0, wfmax = 0;
	char		*wdxfn, fpath[MAX_PATH];
	char		buf[2048], nbuf[SBUF_SIZE], ubuf[SBUF_SIZE];
	char		*s;
	void		*p;
	int			ft, showinfo, ftsz;
	FILE		*fp = NULL;
	int			i, j, n, rc = 0;

	/* parse args */
	wdxfn = NULL;
	*fpath = 0;
	for (i = 1; i < argc; ++i) {
		if ('-' == argv[i][0]) {
			switch (argv[i][1]) {
			case 'f':
				out_fulltext = 1;
				if (1 != sscanf(argv[i]+2, "%u", &max_fulltext)) max_fulltext = 0;
				break;
			default:
				break;
			}
		} else {
			if (!wdxfn) wdxfn = argv[i];
			else if (!*fpath) GetFullPathName(argv[i], sizeof(fpath), fpath, NULL);
		}
	}
	if (!wdxfn) {
		oprintf(stdout, WDXTEST_LOGO);
		oprintf(stdout, WDXTEST_USAGE);
		return 0;
	}
	showinfo = !*fpath;

#ifdef DEBUG
	oprintf(stdout,
		"wdxfn: '%s'\n"
		"fpath: '%s'\n"
		"showinfo: %d, out_fulltext: %d, max_fulltext: %d\n",
		wdxfn ? wdxfn : "--NULL--",
		fpath, showinfo, out_fulltext, max_fulltext);
#endif

	/* load plugin */
	if(!(hwdx = LoadLibrary(wdxfn))) {
		oprintf(stdout, "Failed loading plugin %s.\n", wdxfn);
		rc = WTE_NO_WCX;
		goto stop;
	} else {
		if (showinfo) oprintf(stdout, "Plugin %s loaded.\n", wdxfn);
	}

	/* detect string */
	if (!(pContentGetDetectString = (void *) GetProcAddress(hwdx, "ContentGetDetectString"))) {
		if (showinfo) oprintf(stdout, "Plugin does not imlement ContentGetDetectString()\n");
	} else {
		pContentGetDetectString(buf, sizeof(buf)-1);
		if (showinfo) oprintf(stdout, "Detect string: %s\n", buf);
	}
	/* plugin unloadig */
	if (!(pContentPluginUnloading = (void *) GetProcAddress(hwdx, "ContentPluginUnloading"))) {
		if (showinfo) oprintf(stdout, "Plugin does not imlement ContentPluginUnloading()\n");
	}
	/* settings */
	if (!(pContentSetDefaultParams = (void *) GetProcAddress(hwdx, "ContentSetDefaultParams"))) {
		if (showinfo) oprintf(stdout, "Plugin does not imlement ContentSetDefaultParams()\n");
	} else {
		ContentDefaultParamStruct	dps;

		dps.size = sizeof(ContentDefaultParamStruct);
		dps.PluginInterfaceVersionLow = 4;
		dps.PluginInterfaceVersionHi = 1;
		strcpy(dps.DefaultIniName, argv[0]);
		for (s = dps.DefaultIniName; *s; ++s) ;
		for ( ; s > dps.DefaultIniName && *s != '.'; --s) ;
		if (*s == '.') *s = 0;
		strcat(dps.DefaultIniName, ".ini");
		if (showinfo) oprintf(stdout, "INI file name: %s\n", dps.DefaultIniName);
		pContentSetDefaultParams(&dps);
	}

	/* mandatory functions */
	if (!(pContentGetSupportedField = (void *) GetProcAddress(hwdx, "ContentGetSupportedField"))) {
		oprintf(stdout, "Error: plugin does not imlement mandatory ContentGetSupportedField()\n");
		rc = WTE_NO_PROC;
		goto stop;
	}
	if (!(pContentGetValue = (void *) GetProcAddress(hwdx, "ContentGetValue"))) {
		oprintf(stdout, "Error: plugin does not imlement mandatory ContentGetValue()\n");
		rc = WTE_NO_PROC;
		goto stop;
	}

	if (showinfo) oprintf(stdout, "Plugin fields:\n");
	do {
		ft = pContentGetSupportedField(wfcnt, nbuf, ubuf, SBUF_SIZE);
		if (ft_nomorefields == ft) {
			if (showinfo) oprintf(stdout, "-- no more fields. Number of fields: %d\n", wfcnt);
			break;
		}
		/* show info about field */
		switch (ft) {
			case ft_numeric_32:			s = "ft_numeric_32"; break;
			case ft_numeric_64:			s = "ft_numeric_64"; break;
			case ft_numeric_floating:	s = "ft_numeric_floating"; break;
			case ft_date:				s = "ft_date"; break;
			case ft_time:				s = "ft_time"; break;
			case ft_boolean:			s = "ft_boolean"; break;
			case ft_multiplechoice:		s = "ft_multiplechoice"; break;
			case ft_string:				s = "ft_string"; break;
			case ft_fulltext:			s = "ft_fulltext"; break;
			case ft_datetime:			s = "ft_datetime"; break;
		}
		if (showinfo) {
			oprintf(stdout, "%2d: %s: %s", wfcnt, nbuf, s);
			if (ubuf[0]) oprintf(stdout, " --> %s\n", ubuf);
			else if (!strcmp("-", nbuf)) oprintf(stdout, " --> menu separator (?)\n");
			else oprintf(stdout, "\n");
		}
		/* store info about field */
		if (wfcnt >= wfmax) {
			if (!wfmax) n = 2; else n = 2 * wfmax;
			if (p = realloc(wf, n * sizeof(WDX_FIELD))) {
				wfmax = n;
				wf = p;
			} else {
				oprintf(stdout, "realloc() error at %d\n", __LINE__);
				rc = WTE_NO_MEM;
				goto stop;
			}
		}
		wf[wfcnt].type = ft;
		if (s = malloc(strlen(nbuf)+1)) strcpy(s, nbuf);
		wf[wfcnt].name = s;
		if (ubuf[0] && ft_multiplechoice != ft) {
			if (s = malloc(strlen(ubuf)+1)) strcpy(s, ubuf);
			wf[wfcnt].units = s;
		} else {
			wf[wfcnt].units = NULL;
		}
		++wfcnt;
	} while (ft_nomorefields != ft);
	if (wfcnt < 1) {
		oprintf(stdout, "No usable fields.\n");
		rc = WTE_NO_FIELDS;
		goto stop;
	}

	/* check if ft_fulltext is at the END of the field list */
	for (i = 0; i < wfcnt; ++i) {
		if (ft_fulltext == wf[i].type) {
			for (j = i + 1; j < wfcnt; ++j) {
				if (ft_fulltext != wf[j].type) {
					oprintf(stdout, "Warning: field '%s' (%d) with 'ft_fulltext' type is not at the END of the field list\n", wf[i].name, i);
					break;
				}
			}
		}
	}
	/* check field & unit names
	 * The field may not contain the following chars: . (dot) | (vertical line) : (colon).
	 */
	for (i = 0; i < wfcnt; ++i) {
		char	*s, illch[8];

		/* field name */
		strcpy(illch, ".|:");
		for (s = illch; *s; ++s) {
			if (strchr(wf[i].name, *s))
				oprintf(stdout, "Warning: field (%d) name '%s' contains illegal char '%c'\n", i, wf[i].name, *s);
		}
		/* unit name */
		if (wf[i].units) {
			strcpy(illch, ".:");
			for (s = illch; *s; ++s) {
				if (strchr(wf[i].name, *s))
					oprintf(stdout, "Warning: field's (%d) unit name '%s' contains illegal char '%c'\n", i, wf[i].units, *s);
			}
		}
	}

	if (*fpath) {
		oprintf(stdout, "File '%s'\n", fpath);
		for (i = 0; i < wfcnt; ++i) {
			char	*t;

			oprintf(stdout, "%2d: %s:", i, wf[i].name); fflush(stdout);
			if (ft_fulltext == wf[i].type) {
				oprintf(stdout, " ft_fulltext:", i, wf[i].name);
				if (!out_fulltext) {
					oprintf(stdout, " -- skip (use -f switch).\n");
				} else {
					oprintf(stdout, "\n--\n");
					ftsz = 0;
					do {
						memset(buf, 0, sizeof(buf));
						ft = pContentGetValue(fpath, i, ftsz, buf, sizeof(buf), 0);
						if (ft_fieldempty == ft) {
							oprintf(stdout, "\n-- ft_fieldempty --> stop.\n");
							break;
						}
						if (ft_fulltext != ft) {
							oprintf(stdout, "\n-- unexpected resulting field type ft_* (%d) --> stop.\n", ft);
							break;
						}
						if (!max_fulltext || (ftsz + strlen(buf)) < (unsigned)max_fulltext) {
							_setmode(_fileno(stdout), _O_BINARY);
							fputs(buf, stdout); fflush(stdout);
							_setmode(_fileno(stdout), _O_TEXT);
						} else {
							buf[max_fulltext-ftsz] = 0;
							ftsz = max_fulltext;

							_setmode(_fileno(stdout), _O_BINARY);
							fputs(buf, stdout); fflush(stdout);
							_setmode(_fileno(stdout), _O_TEXT);
							pContentGetValue(fpath, i, -1, buf, sizeof(buf), 0);
							oprintf(stdout, "\n");
							break;
						}
						ftsz += strlen(buf);
					} while (1);
					oprintf(stdout, "-- %d bytes written.\n", ftsz);
				}
			} else if (!strcmp("-", wf[i].name)) {
				oprintf(stdout, " -- menu separator (?).\n");
				continue;
			} else {
				if (wf[i].units) {
					oprintf(stdout, "\n"); fflush(stdout);
					for (s = wf[i].units, j = 0; *s; ) {
						for (t = s; *t && *t != '|'; ++t) ;
						strncpy(ubuf, s, t - s);
						ubuf[t - s] = '\0';
						memset(buf, 0, sizeof(buf));
						oprintf(stdout, "%10s: ", ubuf);
						ft = pContentGetValue(fpath, i, j, buf, sizeof(buf), 0);
						print_fval(ft, buf);
						s = t;
						if (*s) ++s;
						++j;
					}
				} else {
					oprintf(stdout, " ");
					memset(buf, 0, sizeof(buf));
					ft = pContentGetValue(fpath, i, 0, buf, sizeof(buf), 0);
					print_fval(ft, buf);
				}
			}
		}
	}

stop:
	if (wf) {
		for (i = 0; i < wfcnt; ++i) {
			if (wf[i].name) free(wf[i].name);
			if (wf[i].units) free(wf[i].units);
		}
		free(wf);
	}
	if (hwdx) {
		if (pContentPluginUnloading) pContentPluginUnloading();
		FreeLibrary(hwdx);
	}
	if (fp) fclose(fp);
	return rc;
}
